<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder; // added by ss
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Quotation extends Model
{
    const ZEROES = 5;
    const QUOTATION = 'Quotation';
    const VOUCHER = 'Voucher';

    const TYPE_INVENTORY = 'inventory';
    const TYPE_BESPOKE_AND_REPAIR = 'bespoke_and_repair';
    
    const TABLENAME = "quotations";
    const IMAGE_PATH = "img/quotations";
    const TEMP_PATH = "img/temp/";

    const PDF_PATH = "pdf/quotations";

    use HasFactory;

    use \Kyslik\ColumnSortable\Sortable;

    protected $sortable = [	'id', 'quotation_no', 'staff_id', 'staff_name',	'customer_id', 'customer_name',	'customer_email', 'country_code', 'customer_phone',	'reference_date', 'reference_time',	'sub_total', 'gst_percent', 'gst_amount', 'total', 'created_at' ];

    protected $fillable = [	'quotation_no', 'staff_id', 'staff_name', 'customer_id', 'customer_name', 'customer_email', 'country_code', 'customer_phone',	'reference_date', 'reference_time',	'sub_total', 'gst_percent', 'gst_amount', 'total', 'remarks', 'layaway' ];

    protected $appends = [ 'phone_with_code' ];

    protected static function boot()
    {
        parent::boot();

        Quotation::creating(function ($model) {
            $status = 0;
            try{
                if(empty($model->reference_date)){
                    $model->reference_date = date('Y-m-d');
                }

                if(empty($model->reference_time)){
                    $model->reference_time = date('H:i:s');
                }
                $status = 1;
            } catch(\Exception $e){
                $status = 0;
                Log::error('message error in static::creating on Quotation Model :'. $e->getMessage());
            }
        });

        Quotation::retrieved(function ($model) {
            if(empty($model->reference_date)){
                $model->reference_date = date('Y-m-d', strtotime($model->created_at));
            }

            if(empty($model->reference_time)){
                $model->reference_time = date('H:i:s', strtotime($model->created_at));
            }
        });
    }

    public function staff()
    {
        return $this->belongsTo(Admin::class, 'staff_id', 'id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id', 'id');
    }

    public function quotation_item()
    {
        return $this->hasMany(QuotationItem::class, 'quotation_id', 'id');
    }

    public function quotation_loose_piece()
    {
        return $this->hasMany(QuotationLoosePiece::class, 'quotation_id', 'id');
    }

    public function quotation_item_image()
    {
        return $this->hasManyThrough('App\Models\QuotationItemImage', 'App\Models\QuotationItem', 'quotation_id', 'quotation_item_id', 'id')->first();
    }

    public function invoice()
    {
        return $this->hasOne(Invoice::class, 'quotation_id', 'id');
    }

    //-------------------------------------------------------------------
    public function scopeToday($query)
    {
        $query->whereDate('created_at', date('Y-m-d'));
    }

    public function scopeTotal($query)
    {
        // $query->where('status', self::ACTIVE);
        $query->whereNotNull('quotation_no');
    }

    public static function specificQuotationsCountInRange($specification, $startDate, $endDate)
    {
        try {
            $dynamicScope = $specification.'()'; 
            $count = Quotation::{$specification}()->whereBetween('created_at', [$endDate, $startDate])->count(); 
            return $count;
        } catch (\Exception $err) {
            Log::error('Error in specificQuotationsCountInRange on Quotation Model :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
       
    //-------------------------------------------------------------------

    // public function getPhoneWithCodeAttribute()
    // {
    //     return isset($this->customer_phone) ? $this->customer_phone : '-';
    // }

    public function getPhoneWithCodeAttribute()
    {
        return isset($this->customer_phone) ? ($this->country_code ? $this->country_code.' '.$this->customer_phone : $this->customer_phone) : ( ($this->customer) ? $this->customer->customer_phone : '-');
    }
    
}
