<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\JobType;
use File;
use Log;

class Job extends Model
{
    use HasFactory;

    use \Kyslik\ColumnSortable\Sortable;

    use SoftDeletes;

    const ZEROES = 5;
    const JOB = 'Job';
    const VOUCHER = 'Voucher';

    const ACTIVE = "Active";
    const INACTIVE = "Inactive";
    const ARCHIVED = "Archived";

    const NOT_ASSIGNED = "Not Assigned";
    const NOT_STARTED = "Not Started";
    const CRAFTING = "Crafting";
    const COMPLETED = "Completed";
    const RECEIVED = "Received";
    const DELAYED = "Delayed";
    const REJECTED = "Rejected";

    const COLUMBIA_BLUE = "columbia_blue";
    const BLACK = "black";
    const EARTH_YELLOW = "earth_yellow";
    const UNITED_NATIONS_BLUE = "united_nation_blue";
    const MINT = "mint";
    const BEGONIA = "begonia";
    const REDD = "red1";

    const TABLE = 'table';
    const CALENDAR = 'calendar';
    const BESPOKE = "Bespoke";
    const REPAIR = "Repair";
    const TABLENAME = "jobs";

    const JOB_STATUS_COLOR_ARR = [ self::NOT_ASSIGNED=>'#cbd5e1',  self::NOT_STARTED=>'#000', self::CRAFTING=>'#e4a061', self::COMPLETED=>'#6183e4', self::RECEIVED=>'#37b981', self::DELAYED=>'#f77171', self::REJECTED=>'#ff0000' ];

    const JOB_STATUS_CLASS_ARR = [ self::NOT_ASSIGNED=>self::COLUMBIA_BLUE,  self::NOT_STARTED=>self::BLACK, self::CRAFTING=>self::EARTH_YELLOW, self::COMPLETED=>self::UNITED_NATIONS_BLUE, self::RECEIVED=>self::MINT, self::DELAYED=>self::BEGONIA, self::REJECTED=>self::REDD ];

    const JOB_STATUS_DUMMY_CLASS_ARR = [ self::NOT_ASSIGNED=>'blue',  self::NOT_STARTED=>'red', self::CRAFTING=>'green', self::COMPLETED=>'blue', self::RECEIVED=>'disabled', self::DELAYED=>'blue', self::REJECTED=>'red1' ];

    const IMAGE_PATH = "img/jobs";
    const TEMP_PATH = "img/temp/";
    const DRAWING_PATH = "img/drawing_images/";
    const THUMB_IMAGE_PATH = 'img/jobs/thumb';
    const THUMB_SIZE = 70;

    public $sortable = [ 'id', 'job_type_id','item_code','category_id', 'job_status','estimate_completion','name' ];

    protected $fillable = [ 'job_type_id','item_code', 'staff_id','worksmith_id', 'category_id', 'estimate_completion','earing_pin','gold_weight', 'gold_cost', 'craftsmanship_cost', 'polishing',  'job_status', 'customer_id', 'date','description', 'gold_color_id', 'size', 'is_sample_provided', 'sample_description', 'setting_cost', 'backing_cost', 'backing_type', 'backing_size', 'backing_quantity', 'claw', 'status', 'drawing_image', 'customer_signature', 'sales_representative_signature','additional_material','material_quantity','additional_information','is_jewellery_received' ];

    // protected $appends = [ 'drawing_image_url' ];
    // protected $dates = [ 'date' ]; // 'created_at', 'updated_at',

     const MONTH_ARR = ["January","February","March","April","May","June","July","August","September","October","November","December"];

    public static function job_status()
    {
        return [
            self::NOT_ASSIGNED => Self::NOT_ASSIGNED,
            self::NOT_STARTED => Self::NOT_STARTED,
            self::CRAFTING => Self::CRAFTING,
            self::COMPLETED => Self::COMPLETED,
            self::DELAYED => Self::DELAYED,
            self::RECEIVED => Self::RECEIVED,
            // self::REJECTED => Self::REJECTED,
        ];
    }

    public function staff()
    {
        return $this->belongsTo(Admin::class, 'staff_id', 'id');
    }

    public function worksmith()
    {
        return $this->belongsTo(Worksmith::class, 'worksmith_id', 'id');
    }

    public function type()
    {
        return $this->belongsTo(Type::class, 'type_id', 'id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id', 'id');
    }

    public function images()
    {
        return $this->hasMany(JobImage::class, 'job_id', 'id');
    }

    public function first_image()
    {
        return $this->hasOne(JobImage::class, 'job_id', 'id')->latest();
    }

    public function loose_pieces()
    {
        return $this->hasMany(JobLoosePiece::class, 'job_id', 'id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id', 'id');
    }

    public function review_cost()
    {
        return $this->hasOne(ReviewCost::class, 'job_id', 'id');
    }

    public function job_type()
    {
        return $this->belongsTo(JobType::class, 'job_type_id', 'id');
    }

    public function gold_color()
    {
        return $this->belongsTo(GoldColor::class, 'gold_color_id', 'id');
    }

    public static function defaultImage()
    {
        return asset('public/img/default-placeholder.jpg');
    }

    public static function dynamicColor($status)
    {
        return array_key_exists($status, self::JOB_STATUS_CLASS_ARR) ? self::JOB_STATUS_CLASS_ARR[$status] : '';
    }

    //-------------------------------------------------------------------
    public function scopeMonthly($query)
    {
        $query->whereMonth('created_at', date('m'));
    }

    public function scopeTotal($query)
    {
        $query->where('status', self::ACTIVE);
    }

    public function scopeBespoke($query)
    {
        $bespokeTypeId = JobType::getTypeIdByName(JobType::BESPOKE);
        $query->where('job_type_id', $bespokeTypeId);
    }

    public function scopeRepair($query)
    {
        $repairTypeId = JobType::getTypeIdByName(JobType::REPAIR);
        $query->where('job_type_id', $repairTypeId);
    }

    public function scopeNotAssigned($query)
    {
        $query->where('job_status', self::NOT_ASSIGNED);
    }

    public function scopeNotStarted($query)
    {
        $query->where('job_status', self::NOT_STARTED);
    }

    public function scopeCrafting($query)
    {
        $query->where('job_status', self::CRAFTING);
    }

    public function scopeCompleted($query)
    {
        $query->where('job_status', self::COMPLETED);
    }

    public function scopeReceived($query)
    {
        $query->where('job_status', self::RECEIVED);
    }

    public function scopeDelayed($query)
    {
        $query->where('job_status', self::DELAYED);
    }

    // public function getDrawingImageUrlAttribute($value){
    //     if (!empty($value) && File::exists(public_path(Self::DRAWING_PATH).$value)){
    //         return asset('public/img/drawing_images/'.$value);
    //     }else{
    //         return asset('public/img/default-placeholder.jpg');
    //     }
    // }

    public static function specificJobsCountInRange($specification, $startDate, $endDate)
    {
        try {
            $dynamicScope = $specification.'()';
            $count = Job::{$specification}()->whereBetween('created_at', [$endDate, $startDate])->count();
            return $count;
        } catch (\Exception $err) {
            Log::error('Error in specificJobsCountInRange on Job Model :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    //-------------------------------------------------------------------

    public static function getItemCodeArray()
    {
        $result ="";
        $job = Job::whereHas('review_cost', function($q)
        {
          $q->where('total','!=', 0);
        })->pluck('item_code')->toArray();

        if($job){
            $result = $job;
        }else{
            $result = " ";
        }
        return $result;
    }

    public static function getJobByItemCode($item_code){
        $job =Job::with(['category', 'review_cost', 'gold_color'])->where('item_code', $item_code)->first();
        if($job){
            return $job;
        }else{
            return  $job = null;
        }
    }

}

