<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder; // added by ss
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Invoice extends Model
{
    const ZERO = 0;
    const ONE = 1;
    const _ZERO = '0';
    const _ONE = '1';

    const PENDING = "0";
    const IN_PROGRESS = "1";
    const COMPLETED = "2";
    const FAILED = "3";
    
    const ZEROES = 5;
    const INVOICE = 'Invoice';

    const PDF_PATH = "pdf/invoices";

    use HasFactory;

    use \Kyslik\ColumnSortable\Sortable;

    public $sortable = [ 'id', 'invoice_no', 'quotation_id', 'staff_id', 'staff_name', 'customer_id', 'customer_name', 'customer_email', 'country_code', 'customer_phone', 'reference_date', 'reference_time', 'sub_total', 'gst_percent', 'gst_amount', 'total', 'paid_amount', 'balance', 'due', 'refunded_amount', 'payment_status', 'remarks', 'layaway', 'created_at' ];
    
    protected $fillable = [ 'invoice_no', 'quotation_id', 'staff_id', 'staff_name', 'customer_id', 'customer_name', 'customer_email', 'country_code', 'customer_phone', 'reference_date', 'reference_time', 'sub_total', 'gst_percent', 'gst_amount', 'total', 'paid_amount', 'balance', 'due', 'is_refund', 'refunded_amount', 'payment_status', 'remarks', 'layaway' ];

    protected $appends = [ 'phone_with_code' ];

    protected static function boot()
    {
        parent::boot();

        Invoice::creating(function ($model) {
            $status = 0;
            try{
                $model->reference_date = date('Y-m-d');
                $model->reference_time = date('H:i:s');
                $status = 1;
            } catch(\Exception $e){
                $status = 0;
                Log::error('message error in static::creating on Invoice Model :'. $e->getMessage());
            }
        });

        Invoice::retrieved(function ($model) {
            if(empty($model->reference_date)){
                $model->reference_date = date('Y-m-d', strtotime($model->created_at));
            }

            if(empty($model->reference_time)){
                $model->reference_time = date('H:i:s', strtotime($model->created_at));
            }
        });
    }

    public function staff()
    {
        return $this->belongsTo(Admin::class, 'staff_id', 'id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id', 'id');
    }

    public function invoice_item()
    {
        return $this->hasMany(InvoiceItem::class, 'invoice_id', 'id');
    }

    public function quotation()
    {
        return $this->belongsTo(Quotation::class, 'quotation_id', 'id');
    }

    public function transaction()
    {
        return $this->hasMany(Transaction::class, 'invoice_id', 'id');
    }

    //-------------------------------------------------------------------
    public function scopeToday($query)
    {
        $query->whereDate('created_at', date('Y-m-d'));
    }

    public function scopeTotal($query)
    {
        // $query->where('status', self::ACTIVE);
        $query->whereNotNull('invoice_no');
    }

    public function scopeConvertedInvoices($query)
    {
        $query->has('quotation');
    }

    public static function specificInvoicesCountInRange($specification, $startDate, $endDate)
    {
        try {
            $dynamicScope = $specification.'()'; 
            $count = Invoice::{$specification}()->whereBetween('created_at', [$endDate, $startDate])->count(); 
            return $count;
        } catch (\Exception $err) {
            Log::error('Error in specificInvoicesCountInRange on Invoice Model :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
    //-------------------------------------------------------------------

    // public function getPhoneWithCodeAttribute()
    // {
    //     return isset($this->customer_phone) ? $this->customer_phone : '-';
    // }

    public function getPhoneWithCodeAttribute()
    {
        return isset($this->customer_phone) ? ($this->country_code ? $this->country_code.' '.$this->customer_phone : $this->customer_phone) : ( ($this->customer) ? $this->customer->customer_phone : '-');
    }

    public static function checkInvoiceByQuotation($quotation_id)
    {
      $invoice  = Invoice::where('quotation_id',$quotation_id)->first();
      if(!empty($invoice)){
        return true;
      }else{
        return false;
      }
    }
}
