<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Components\MultipleDropzoneComponent;
use App\Http\Controllers\Components\QuotationComponent;
use App\Http\Controllers\Components\CustomerComponent;
use App\Http\Controllers\Components\PDFComponent;
use Illuminate\Http\Request;
use App\Http\Requests\Quotations\QuotationsRequest;
use App\Http\Requests\Quotations\CreateInvoicesRequest;
use App\Http\Requests\Quotations\SendToJobRequest;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Mail\QuotationPDFMail;
use App\Helpers\CommonHelper;
use App\Models\QuotationItemImage;
use App\Models\QuotationItem;
use App\Models\InvoiceItem;
use App\Models\Quotation;
use App\Models\GoldColor;
use App\Models\Jewellery;
use App\Models\Category;
use App\Models\Invoice;
use App\Models\Setting;
use App\Models\JobType;
use App\Models\ReviewCost;
use App\Models\User;
use App\Models\Job;
use Mail;
use Auth;
use File;
use PDF;
use Log;
use DB;


class QuotationsController extends Controller
{
    const ZEROES = 5;
    const QUOTATION = 'Quotation';
    const INVOICE = 'Invoice';
    const JOB = 'Job';

    function __construct(MultipleDropzoneComponent $multipleDropzone)
    {
        $this->MultipleDropzoneComponent = $multipleDropzone;
        $this->QuotationComponent = new QuotationComponent();
        $this->CustomerComponent = new CustomerComponent();
        $this->PDFComponent =  new PDFComponent();
    }

    public function create()
    {
        try {
            return view('frontend.quotations.create');
        } catch (\Exception $err) {
            Log::error('Error in create on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function quotationsStore(QuotationsRequest $request)
    {
        DB::beginTransaction();
        try {
            if(!empty((float)$request->subtotal)){
                $res = [];
                $inputData = $request->except(['qut']);
                $customer = $request->only(['customer_phone', 'customer_name', 'customer_email']);
                $customerRes = $this->CustomerComponent->addOrupdateCustomer($customer);

                if ($customerRes['status'] == 1) {

                    $phoneInputRes = $this->separateDetailsFromPhoneInput($request['customer_phone']);
                    $inputData['staff_name'] = Auth::guard('web')->user()->fullname;
                    $inputData['staff_id'] = Auth::guard('web')->user()->id;
                    $inputData['country_code'] = $phoneInputRes['country_code'];
                    $inputData['customer_phone'] = $phoneInputRes['phone'];
                    $inputData['customer_id'] = $customerRes['data'];
                    $currentTimestamp = time();
                    if (empty($invoiceData['reference_date'])) {
                        $inputData['reference_date'] = date('Y-m-d', $currentTimestamp);
                    }
                    if (empty($invoiceData['reference_time'])) {
                        $inputData['reference_time'] = date('H:i:s', $currentTimestamp);
                    }

                    $subtotal =  $request->subtotal;
                    $settings  = Setting::price_settings();
                    $inputData['sub_total'] = $subtotal;

                    $inputData['gst_percent'] = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                    $inputData['gst_amount'] = ($subtotal *  $inputData['gst_percent'])/ 100 ;
                    $inputData['total'] =  ($inputData['sub_total'] + $inputData['gst_amount']) ;

                    $quotation = new Quotation();
                    $quotation->fill($inputData);
                    if($quotation->save()){

                        $updateQuotationNumberRes = $this->_updateQuotationNumber(self::QUOTATION, $quotation->id);
                        if ($updateQuotationNumberRes['status'] == 1) {
                            $quotationData =   $request->qut;
                            foreach ($request->qut as $key => $qut_item) {
                                if (@$qut_item[QuotationItem::INVENTORY]) {
                                    $inventData = $qut_item[QuotationItem::INVENTORY];
                                    if (array_key_exists("checked", $inventData)) {
                                        if ($inventData['item_code']) {
                                            $jewellery = Jewellery::getJewelleryByItemCode($inventData['item_code']);
                                            $qutation_data['quotation_id'] =  $quotation->id;
                                            $qutation_data['item_code'] =  $jewellery->item_code;
                                            $qutation_data['jewellery_id'] = $jewellery->id;
                                            $qutation_data['type_name'] = QuotationItem::INVENTORY;
                                            $qutation_data['category_id'] = $jewellery->category_id;
                                            $qutation_data['gold_color_id'] = $jewellery->gold_color_id ?? null;
                                            $qutation_data['description'] = $jewellery->description ?? null;
                                            $qutation_data['price'] = $jewellery->review_cost->cost_price ?? null;
                                            $qutation_data['sub_total'] = $jewellery->review_cost->sub_total ?? null;
                                            $quotation_item = $quotation->quotation_item()->create($qutation_data);
                                        }
                                    }
                                }

                                if (@$qut_item[QuotationItem::BESPOKE]) {
                                    $bespoke_item = $qut_item[QuotationItem::BESPOKE];
                                    if (array_key_exists("checked", $bespoke_item)) {
                                        $bespoke_item['type_name'] = QuotationItem::BESPOKE;
                                        $bespoke_item['sub_total'] = $bespoke_item['price'] ;
                                        $bespoke_item['additional_material'] = $bespoke_item['material'];
                                        $bespoke_item['material_quantity'] = $bespoke_item['quantity'];
                                        $bespoke_item['customer_received'] = $bespoke_item['customer_recieved'] ?? "0" ;

                                        if(!empty($bespoke_item['illustration'])){
                                            if (!empty($bespoke_item['drawing_textarea_base64'])) {
                                                $drawingImageName = CommonHelper::uploadBase64Image($bespoke_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                                $bespoke_item['drawing_image'] = $drawingImageName;
                                            }
                                        }

                                        $quotation_item = $quotation->quotation_item()->create($bespoke_item);
                                          if (!empty($quotation_item)) {
                                            if(!empty($bespoke_item['illustration'])){
                                            if (!empty($bespoke_item['images'])) {
                                                $this->addQuotationImages($quotation_item, $bespoke_item['images']);
                                            }
                                        }
                                        }
                                    }
                                }

                                if (@$qut_item[QuotationItem::REPAIR]) {
                                    $repair_item = $qut_item[QuotationItem::REPAIR];
                                    if (array_key_exists("checked", $repair_item)) {
                                        $repair_item['type_name'] = QuotationItem::REPAIR;
                                        $repair_item['sub_total'] = $repair_item['price'] ;
                                        $repair_item['additional_material'] = $repair_item['material'];
                                        $repair_item['material_quantity'] = $repair_item['quantity'];
                                        $repair_item['customer_received'] = $repair_item['customer_recieved'] ?? "0" ;

                                     if(!empty($bespoke_item['illustration'])){
                                        if (!empty($repair_item['drawing_textarea_base64'])) {
                                            $drawingImageName = CommonHelper::uploadBase64Image($repair_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                            $repair_item['drawing_image'] = $drawingImageName;
                                        }
                                      }
                                        $quotation_item = $quotation->quotation_item()->create($repair_item);

                                      if(!empty($bespoke_item['illustration'])){
                                        if (!empty($quotation_item)) {
                                            if (!empty($repair_item['images'])) {
                                                $this->addQuotationImages($quotation_item, $repair_item['images']);
                                            }
                                        }
                                      }
                                    }
                                }
                            }

                            DB::commit();
                            $res['status'] = 1;
                            $res['message'] = "Quotation Added Successfully!";
                            $res['data'] =  $quotation;
                        } else {
                            $res['status'] = 0;
                            $res['message'] = $updateQuotationNumberRes['message'];
                        }
                    }else{
                        $res['status'] = 0;
                        $res['message'] = "Quotation not added!";
                    }
                } else {
                    $res['status'] = 0;
                    $res['message'] = $customerRes['message'];
                }
            }else{
                $res['status'] = 0;
                $res['message'] = "Please select price!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in quotationsStore on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }


    public function addQuotationImages($quotation_item,$quot_images)
    {
        try {
            $res = [];
            $imagePath = QuotationItemImage::IMAGE_PATH;
            $tmpDirPath = QuotationItemImage::TEMP_PATH;
            $tableName = Quotation::TABLENAME;
            foreach ($quot_images as $val_img){
                $this->MultipleDropzoneComponent->storeImage($val_img, $imagePath, $tmpDirPath, $tableName);
                $bespoke_items = $quotation_item->quotation_item_image()->create([
                    'quotation_id' => $quotation_item->quotation_id,
                    'quotation_item_id' => $quotation_item->id,
                    'image' => $val_img,
                ]);
            }
            $res['status'] = 1;
            $res['message'] = "Quotation images added successfully!";
        } catch (\Exception $err) {
            Log::error('Error in addQuotationImages on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function todaysOrders(Request $request, $tab=null)
    {
        try {
            if($tab == null){
                $tab = QuotationItem::TAB_INVENTORY;
            }else{
                $tab = $tab;
            }

            $todayInventoryCount = QuotationItem::today()->inventory()->count();
            $todayBespokeCount = QuotationItem::today()->bespoke()->count();
            $todayRepairCount = QuotationItem::today()->repair()->count();

            $inventoryOrders = $this->todaysInventoryQuotations($request);
            $bespokeAndRepairOrders = $this->todaysBespokeAndRepairQuotations($request);
            return view('frontend.quotations.index', compact(['tab', 'todayInventoryCount', 'todayBespokeCount', 'todayRepairCount', 'inventoryOrders', 'bespokeAndRepairOrders']));
        } catch (\Exception $err) {
            Log::error('Error in todaysOrders on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function todaysInventoryQuotations($request)
    {
        try{
            $records = Quotation::with(['customer', 'quotation_item'])->whereHas('quotation_item', function ($q) {
                $q->inventory();
            })->withCount(['quotation_item as bespokeAndRepairCount'=>function($q){
                $q->bespokeAndRepair();
            }])->today()->sortable(['id' => 'desc']);

            $this->commonSearch($request, $records);
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return $records;
        }catch(\Exception $err){
            Log::error('Error in todaysInventoryQuotations on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function todaysBespokeAndRepairQuotations($request)
    {
        try{
            $records = Quotation::with(['customer', 'quotation_item'])->whereHas('quotation_item', function ($q) {
                    $q->bespokeAndRepair();
                })->withCount(['quotation_item as inventoryCount'=>function($q){
                    $q->inventory();
                }])->today()->sortable(['id' => 'desc']);

            $this->commonSearch($request, $records);
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return $records;
        }catch(\Exception $err){
            Log::error('Error in todaysBespokeAndRepairQuotations on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function commonSearch($request, $records)
    {
        try {
            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('quotation_no', 'like', '%' . $request->query('search') . '%');

                    $q->orWhereHas('quotation_item', function ($query) use ($request) {
                        $query->where('item_code', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            return $records;
        } catch(\Exception $err){
            Log::error('Error in commonSearch on Front\QuotationsController :' . $err->getMessage());
        }
    }

    public function showItemsList($quotation_id, $type)
    {
        try {
            if($type == Quotation::TYPE_INVENTORY){
                $record = $this->__inventoryItemsList($quotation_id, $type);
            }else if($type == Quotation::TYPE_BESPOKE_AND_REPAIR){
                $record = $this->__bespokeAndRepairItemsList($quotation_id, $type);
            }
            return view('frontend.quotations.quotation_items', compact('quotation_id', 'type', 'record'));
        } catch (\Exception $err) {
            Log::error('Error in showItemsList on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function __inventoryItemsList($quotation_id, $type)
    {
        try {
            return $record = Quotation::with(['customer', 'quotation_item'=>function($q){
                $q->inventory();
            }])->withCount(['quotation_item as bespokeAndRepairCount'=>function($q){
                $q->bespokeAndRepair();
            }])->today()->findOrFail($quotation_id);
        } catch (\Exception $err) {
            Log::error('Error in __inventoryItemsList on Front\QuotationsController :' . $err->getMessage());
        }
    }

    public function __bespokeAndRepairItemsList($quotation_id, $type)
    {
        try {
            return $record = Quotation::with(['customer', 'quotation_item'=>function($q){
                $q->bespokeAndRepair();
            }])->withCount(['quotation_item as inventoryCount'=>function($q){
                $q->inventory();
            }])->today()->findOrFail($quotation_id);
        } catch (\Exception $err) {
            Log::error('Error in __bespokeAndRepairItemsList on Front\QuotationsController :' . $err->getMessage());
        }
    }

    public function printQuotation($quotation_id)
    {
        try {

            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
                return view('frontend.quotations.print.print_quotation', compact('quotation'));
            }else{
                return back()->with('error', $quotationRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printQuotation on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printOrderVoucher($quotation_id)
    {
        try {
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
                return view('frontend.quotations.print.print_order_voucher', compact('quotation'));
            }else{
                return back()->with('error', $quotationRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printOrderVoucher on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printBoth($quotation_id)
    {
        try {
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
                return view('frontend.quotations.print.print_both', compact('quotation'));
            }else{
                return back()->with('error', $quotationRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printBoth on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Quotation::findOrFail($id);
            if ($record->delete()) {
                /*Delete Quotation Images Here*/
                return back()->with(['success' => 'Quotation deleted successfully.']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on Front\QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function convertToInvoice(Request $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $quotation_id = $request->id;
            $quotation = Quotation::with('quotation_item')->findOrFail($quotation_id);
            $quotationData = is_array($quotation) ? $quotation : $quotation->toArray();
            $quotationData['quotation_id'] = $quotation->id;

            $quotationData['paid_amount'] = 0;
            $quotationData['balance'] = 0;
            $quotationData['due'] = $quotationData['total'];

            $invoice = new Invoice();
            $invoice->fill($quotationData);
            $invoice->save();

            if ($invoice->exists()) {
                $updateInvoiceNumberRes = $this->_updateInvoiceNumber(self::INVOICE, $invoice->id);

                if($updateInvoiceNumberRes['status'] == 1){
                    $invoice_id = $invoice->id;
                    $quotationItemData['invoice_id'] = $invoice_id;


                    foreach($quotationData['quotation_item'] as $key => $quotation_item){
                        $invoice_id = $invoice->id;
                        $quotation_item['invoice_id'] = $invoice_id;
                        $invoice_item = new InvoiceItem();
                        $invoice_item->fill($quotation_item);
                        $invoice_item->save();
                    }
                }

                if ($invoice_item->exists()) {
                    DB::commit();
                    $res['status'] = 1;
                    $res['message'] = "Quotation converted to invoice successfully.";
                    $res['data'] = $invoice->load('customer') ?? null;
                } else {
                    $res['status'] = 0;
                    $res['message'] = "Unable to save quotation item into invoice!";
                }
            } else {
                $res['status'] = 0;
                $res['message'] = "Unable to save quotation into invoice!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in convertToInvoice on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
        return response()->json($res);
    }

    public function _updateQuotationNumber($string, $quotation_id)
    {
        $res = [];
        try {
            $record = Quotation::findOrFail($quotation_id);
            $record->quotation_no = getItemCode(self::ZEROES, $quotation_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Quotation Number updated successfully.";
                $res['data'] = $record->quotation_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Quotation Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateQuotationNumber on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function _updateInvoiceNumber($string, $invoice_id)
    {
        $res = [];
        try {
            $record = Invoice::findOrFail($invoice_id);
            $record->invoice_no = getItemCode(self::ZEROES, $invoice_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Invoice Number updated successfully.";
                $res['data'] = $record->invoice_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Invoice Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateInvoiceNumber on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function quotationItemDetail($quotation_item_id)
    {
        $res = [];
        try {
            $record = QuotationItem::with(['jewellery:id,drawing_image', 'jewellery.images'=>function($q){
                $q->select('id', 'jewellery_id', 'image')->orderBY('id')->limit(1);
            }])->findOrFail($quotation_item_id);

            $res['status'] = 1;
            $res['title'] = 'Success';
            $res['message'] = "Quotation item found!";

            $res['image_src'] = quotationItemImage($record) ?? getDefaultImage();
        } catch (\Exception $err) {
            Log::error('Error in quotationItemDetail on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }


    public function sendEmailPDF(Request $request, $quotation_id=null)
    {
        $res = [];
        try{
            if($request->ajax()){
                $quotation_id = $request->record_id;
            }
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $quotationRes['message'];
            }

            $quotationPDFRes = $this->PDFComponent->generateQuotationPDF($quotation);
            if($quotationPDFRes['status'] == 1){
                $quotationPDF = $quotationPDFRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $quotationPDFRes['message'];
            }

            $orderVoucherPDFRes = $this->PDFComponent->generateOrderVoucherPDF($quotation);
            if($orderVoucherPDFRes['status'] == 1){
                $orderVoucherPDF = $orderVoucherPDFRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $orderVoucherPDFRes['message'];
            }

            $mailData = [
                'name'    => $quotation->customer_name ?? $quotation->customer->fullname,
                'email'   => $quotation->customer_email ?? $quotation->customer->email,
                'message' => 'Attach with the quotation pdf and order voucher pdf.',
                'quotationPDF'     => $quotationPDF,
                'orderVoucherPDF'  => $orderVoucherPDF,
            ];

            Mail::to($mailData['email'])->send(new QuotationPDFMail($mailData));

            $res['status'] = 1;
            $res['message'] = 'Quotation and Order Voucher Mail sent successfully!';
        } catch (\Exception $err) {
            Log::error('Error in sendemailPDF on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function createInvoice(CreateInvoicesRequest $request)
    {
        $res = [];
        DB::beginTransaction();
        try{

            $validated = $request->validated();

            $quotationItems = QuotationItem::whereIn('id', $validated['qut_item_ids'])->get();

            $quotation_id = $validated['quotation_id'];
            $quotation = Quotation::with(['quotation_item'=>function($q) use ($validated){
                $q->whereIn('id', $validated['qut_item_ids'])->get();
            }])->withSum('quotation_item','sub_total')->addSelect(DB::raw("'-' as gst_total"))->addSelect(DB::raw("'-' as grand_total"))->findOrFail($quotation_id)->makeHidden(['created_at', 'updated_at']);

            $quotation->quotation_item_sum_sub_total = $quotation->quotation_item_sum_sub_total ?? 0;
            $quotation->gst_total = calculateGST($quotation->quotation_item_sum_sub_total);
            $quotation->grand_total = (float) ($quotation->gst_total + $quotation->quotation_item_sum_sub_total);


            $quotationData = is_array($quotation) ? $quotation : $quotation->toArray();
            $quotationData['quotation_id'] = $quotation->id;

            $quotationData['sub_total'] = $quotation->quotation_item_sum_sub_total;
            $quotationData['gst_amount'] = $quotation->gst_total;
            $quotationData['total'] = $quotation->grand_total;

            $quotationData['paid_amount'] = 0;
            $quotationData['balance'] = 0;
            $quotationData['due'] = $quotationData['total'];
            $quotationData['is_refund'] = 0;
            $quotationData['refunded_amount'] = null;

            $quotationData['reference_date'] = date('Y-m-d');
            $quotationData['reference_time'] = date('H:i:s');

            $invoice = new Invoice();
            $invoice->fill($quotationData);
            $invoice->save();
            if ($invoice->exists()) {
                $invoice_id = $invoice->id;
                $updateInvoiceNumberRes = $this->_updateInvoiceNumber(self::INVOICE, $invoice->id);
                if($updateInvoiceNumberRes['status'] == 1){
                    $quotationItemsColl = collect($quotationItems);
                    $invoiceItemsColl = $quotationItemsColl->map(function ($item, $key) use ($invoice_id) {
                        return [
                            'invoice_id' => $invoice_id,
                            'quotation_item_id' => $item->id,
                            'jewellery_id' => $item->jewellery_id,
                            'job_id' => $item->job_id,
                            'item_code' => $item->item_code,
                            'category_id' => $item->category_id,
                            'gold_color_id' => $item->gold_color_id,
                            'additional_material' => $item->additional_material,
                            'material_quantity' => $item->material_quantity,
                            'additional_info' => $item->additional_information,
                            'description' => $item->description,
                            'customer_received' => $item->customer_received,
                            'price' => $item->price,
                            'valued_at' => $item->valued_at,
                            'sub_total' => $item->sub_total,
                        ];
                    })->all();

                    $invoice_items_data = [];
                    foreach($invoiceItemsColl as $invoiceItem) {
                        $invoice_items_data[] = new InvoiceItem($invoiceItem);
                    }

                    $invoice->invoice_item()->saveMany($invoice_items_data);
                    deleteItemsInJewellery($quotationItems);

                    DB::commit();
                    $res['status'] = 1;
                    $res['message'] = 'Invoice created successfully.';
                    $res['data'] = $invoice->load('customer') ?? null;
                }else{
                    $res['status'] = 0;
                    $res['message'] = $updateInvoiceNumberRes['message'];
                }
            }else{
                $res['status'] = 0;
                $res['message'] = 'Invoice not created.';
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in createInvoice on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }


    public function sendToJobOrder(SendToJobRequest $request)
    {
        $res = [];
        DB::beginTransaction();
        try{

            // $invoice = InvoiceItem::where('quotation_item_id',$request->quotation_id)->get();
            // // dd($invoice);
            // if(!empty($invoice) && count($invoice)==0){
            //     // dd('grr');
            //     $res['status'] = 0;
            //     $res['message'] = "Invoice not exist!";
            // }else{
            $quotationItems = QuotationItem::with(['quotation','quotation_item_image','quotation_loose_piece'])->whereIn('id', $request['qut_item_ids'])->get();
            if(!empty($quotationItems)){

                foreach($quotationItems as $key=>$value){

                    $job_type_id =  JobType::getTypeIdByName($value->type_name);
                    $jobInput['job_type_id'] = $job_type_id;
                    $jobInput['staff_id'] = $value->quotation->staff_id;
                    $jobInput['customer_id'] = $value->quotation->customer_id;
                    $jobInput['category_id'] = $value->category_id;
                    $jobInput['gold_weight'] = $value->gold_weight;
                    $jobInput['gold_cost'] = $value->gold_cost;
                    $jobInput['craftsmanship_cost'] = $value->craftsmanship_cost;
                    $jobInput['polishing'] = $value->polishing;
                    $jobInput['size'] = $value->size;
                    $jobInput['is_sample_provided'] = $value->is_sample_provided;
                    $jobInput['sample_description'] = $value->sample_description;
                    $jobInput['setting_cost'] = $value->setting_cost;
                    $jobInput['earing_pin'] = $value->earing_pin;
                    $jobInput['backing_cost'] = $value->backing_cost;
                    $jobInput['backing_type'] = $value->backing_type;
                    $jobInput['backing_size'] = $value->backing_size;
                    $jobInput['backing_quantity'] = $value->backing_quantity;
                    $jobInput['claw'] = $value->claw;
                    $jobInput['worksmith_id'] = $value->worksmith_id;
                    $jobInput['additional_information'] = $value->additional_information;
                    $jobInput['gold_color_id'] = $value->gold_color_id;
                    $jobInput['additional_material'] = $value->additional_material;
                    $jobInput['material_quantity'] = $value->material_quantity;
                    $jobInput['description'] = $value->description;
                    $jobInput['customer_received'] = $value->customer_received;
                    $jobInput['drawing_image'] = $value->drawing_image;

                    $job = new Job();
                    $job->fill($jobInput);
                    $job->save();
                    if($job->exists()){

                       if($job['drawing_image']){
                            if (File::exists(public_path(QuotationItem::DRAWING_PATH . '/' . $job->drawing_image))) {
                                File::copy(public_path(QuotationItem::DRAWING_PATH . '/' . $job->drawing_image), public_path(Job::DRAWING_PATH . '/' . $job->drawing_image));
                            }
                       }
                        $updateJobItemCodeRes = $this->_updateJobItemCode(self::JOB, $job->id);
                        $updateJob = Job::where('id',$job->id)->first();
                        $updateQuotation = QuotationItem::where('id',$value->id)->update(['job_id'=>$job->id,'item_code'=>$updateJob->item_code]);
                        if ($updateJobItemCodeRes['status'] == 1) {
                            $this->_addReviewCost($value , $job);
                            if(!empty($value->quotation_item_image)){
                                $this->_addJobImages($job,$value->quotation_item_image );
                            }
                            if(!empty($value->quotation_loose_piece)){
                                $this->_addJobLoosePieces($job,$value->quotation_loose_piece);
                            }
                        } else{
                            $res['status'] = 0;
                            $res['message'] = $updateJobItemCodeRes['message'];
                        }
                        DB::commit();
                        $res['status'] = 1;
                        $res['message'] = 'job created successfully!';
                    }else{
                        $res['status'] = 0;
                        $res['message'] = 'job not created!';
                    }

                }
            }
        //    }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in sendToJobOrder on Front\QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function _addReviewCost($inputData, $job)
    {
        try{

            $setting =  Setting::price_settings();
            $review_cost_data = [];
            $cost_price = $inputData['price'];
            $sub_total = $inputData->sub_total;
            $gst_percent  =  $setting['gst_percent'];
            $gst_amount = ($sub_total * $gst_percent) /100;
            $total = $sub_total + $gst_amount;

            $review_cost_data['cost_price'] = $cost_price;
            $review_cost_data['sub_total'] = $sub_total;
            $review_cost_data['gst_percent'] = $gst_percent;
            $review_cost_data['gst_amount'] = $gst_amount;
            $review_cost_data['total'] = $total;
            $job_review =  $job->review_cost()->create($review_cost_data);

            if($job_review){
                return true;
            }else{
                return false;
            }
        } catch (\Exception $err) {
            Log::error('Error in _addReviewCost on Frontend/JobsController :' . $err->getMessage());
            return false;
        }
    }
    public function _addJobImages($job,$job_images)
    {
        try {
            $res = [];
            foreach ($job_images as $val_img){


                $store_images= $job->images()->create([
                    'job_id' => $job->id,
                    'image' => $val_img->image,
                ]);
                if (File::exists(public_path(QuotationItemImage::IMAGE_PATH . '/' . $val_img->image))) {
                    File::copy(public_path(QuotationItemImage::IMAGE_PATH . '/' . $val_img->image), public_path(Job::IMAGE_PATH . '/' . $val_img->image));
                }
            }
            $res['status'] = 1;
            $res['message'] = "Job images added successfully!";
        } catch (\Exception $err) {
            Log::error('Error in addJobImages on Frontend/QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function _updateJobItemCode($string, $job_id)
    {
        $res = [];
        try {
            if ($string == self::JOB) {
                $record = Job::findOrFail($job_id);
            } else {
                $record = Jewellery::findOrFail($job_id);
            }

            $record->item_code = getItemCode(self::ZEROES, $job_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Item Code updated successfully.";
                $res['data'] = $record->item_code;
            } else {
                $res['status'] = 0;
                $res['message'] = "Item Code not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateItemCode on Frontend/JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function _addJobLoosePieces($job, $quotationLoosePieces)
    {
    $res = [];
    try {

        foreach ($quotationLoosePieces as $value) {
            if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                $job_loosePieses['stock_no'] = $value['stock_no'];
                $job_loosePieses['weight'] = $value['weight'];
                $job_loosePieses['unit_price'] = $value['unit_price'];
                $job_loosePieses['price'] = $value['price'];
                $job->loose_pieces()->create($job_loosePieses);
            }
        }
        $res['status'] = 1;
        $res['message'] = "Job LoosePieces added successfully!";
    } catch (\Exception $err) {
        Log::error('Error in _addJobLoosePieces on Frontend/QuotationsController :' . $err->getMessage());
        $res['status'] = 0;
        $res['message'] = $err->getMessage();
    }
    return $res;
    }

}

