<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Components\CustomerComponent;
use App\Http\Controllers\Components\InvoiceComponent;
use App\Http\Controllers\Components\PDFComponent;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use App\Http\Requests\Invoices\InvoicesRequest;
use App\Http\Requests\Invoices\UpdatePaymentsRequest;
use App\Http\Requests\Invoices\UpdatePaymentAmountsRequest;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Mail\InvoicePDFMail;
use App\Models\Transaction;
use App\Models\InvoiceItem;
use App\Models\Jewellery;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Job;
use App\Models\Setting;
use Mail;
use File;
use Auth;
use PDF;
use Log;
use DB;


class InvoicesController extends Controller
{
    const ZEROES = 5;
    const INVOICE = 'Invoice';

    function __construct()
    {
        $this->CustomerComponent = new CustomerComponent();
        $this->InvoiceComponent = new InvoiceComponent();
        $this->PDFComponent = new PDFComponent();
    }

    public function create()
    {
        try {
            return view('frontend.invoices.create');
        } catch (\Exception $err) {
            Log::error('Error in create on Front\InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(InvoicesRequest $request)
    {
        DB::beginTransaction();
        try {
            if(empty((float)$request->subtotal)){
                $res['status'] = 0;
                $res['message'] = "Please select price!";
            } else{
                $res = [];
                $inputData = $request->except(['inv']);
                $customer = $request->only(['customer_phone', 'customer_name', 'customer_email']);
                $customerRes = $this->CustomerComponent->addOrupdateCustomer($customer);

                if ($customerRes['status'] == 1) {
                    $phoneInputRes = $this->separateDetailsFromPhoneInput($request['customer_phone']);
                    $inputData['country_code'] = $phoneInputRes['country_code'];
                    $inputData['customer_phone'] =$phoneInputRes['phone'];
                    $inputData['customer_id']  =$customerRes['data'];
                    $inputData['staff_id'] = Auth::guard('web')->user()->id;
                    $inputData['staff_name'] = Auth::guard('web')->user()->fullname;

                    $currentTimestamp = time();
                    if (empty($invoiceData['reference_date'])) {
                        $inputData['reference_date'] = date('Y-m-d', $currentTimestamp);
                    }
                    if (empty($invoiceData['reference_time'])) {
                        $inputData['reference_time'] = date('H:i:s', $currentTimestamp);
                    }
                   
                    $subtotal =  $request->subtotal;
                    $settings  = Setting::price_settings();
                    $inputData['sub_total'] = $subtotal;
                    $inputData['gst_percent'] = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                    $inputData['gst_amount'] = ($subtotal *  $inputData['gst_percent'])/ 100 ;
                    $inputData['total'] =  ($inputData['sub_total'] + $inputData['gst_amount']) ;

                    $inputData['paid_amount'] = 0;
                    $inputData['balance'] = 0;
                    $inputData['due'] = $inputData['total'];
                    $inputData['payment_status'] = Invoice::PENDING;
                  
                    $invoice = new Invoice();
                    $invoice->fill($inputData);
                    $invoice->save();
                
                    if ($invoice->exists()) {
                        $updateInvoiceNumberRes = $this->_updateInvoiceNumber(self::INVOICE, $invoice->id);
                        if ($updateInvoiceNumberRes['status'] == 1) {
                            $invoice_item = [];
                            foreach ($request->inv as $key => $inv_item) {
                                foreach ($inv_item as $index => $val) {
                                    if ($index == InvoiceItem::INVENTORY) {
                                        if (array_key_exists("checked", $val)) {
                                            $jewellery = Jewellery::getJewelleryByItemCode($val['item_code']);
                                            if ($jewellery) {
                                                $inven_data = [];
                                                $inven_data['invoice_id'] = $invoice->id;
                                                $inven_data['item_code'] = $jewellery->item_code;
                                                $inven_data['additional_material'] = $val['additional_material'];
                                                $inven_data['jewellery_id'] = $jewellery->id;
                                                $inven_data['category_id'] = $jewellery->category_id;
                                                $inven_data['gold_color_id'] = $jewellery->gold_color_id ?? null;
                                                $inven_data['description'] = $jewellery->description ?? null;
                                                $inven_data['price'] = $jewellery->review_cost->cost_price ?? null;
                                                $inven_data['sub_total'] = $jewellery->review_cost->sub_total ?? null;
                                                $inventory_item = new InvoiceItem($inven_data);
                                                array_push($invoice_item, $inventory_item);
                                            } else {
                                                $res['status'] = 0;
                                                $res['message'] = 'Item_code not exist!';
                                            }
                                        }
                                    }
                                    if ($index == InvoiceItem::BESPOKE || $index == InvoiceItem::REPAIR) {
                                        if (array_key_exists("checked", $val)) {
                                            $job = Job::getJobByItemCode($val['item_code']);
                                            if ($job) {
                                                $bes_rep_data = [];
                                                $bes_rep_data['invoice_id'] =  $invoice->id;
                                                $bes_rep_data['item_code'] =  $job->item_code;
                                                $bes_rep_data['additional_material'] = $val['additional_material'];
                                                $bes_rep_data['job_id'] =  $job->id;
                                                $bes_rep_data['category_id'] = $job->category_id;
                                                $bes_rep_data['gold_color_id'] = $job->gold_color_id ?? null;
                                                $bes_rep_data['description'] = $job->description ?? null;
                                                $bes_rep_data['price'] = $jewellery->review_cost->sub_total ?? null;
                                                $bes_rep_data['sub_total'] =  $job->review_cost->sub_total ?? null;
                                                $bespoke_repair_item = new InvoiceItem($bes_rep_data);
                                                array_push($invoice_item, $bespoke_repair_item);
                                            } else {
                                                $res['status'] = 0;
                                                $res['message'] = 'Item Code not exist!';
                                            }
                                        }
                                    }
                                }
                            }
                            $saveInvoiceItem = $invoice->invoice_item()->saveMany($invoice_item);
                        
                            if ($saveInvoiceItem) {
                                deleteItemsInJewellery($saveInvoiceItem);
                                DB::commit();
                                $res['status'] = 1;
                                $res['message'] = 'Invoice Item added successfully!';
                                $res['data'] = $invoice->load('customer'); 
                            } else {
                                $res['status'] = 0;
                                $res['message'] = 'Invoice Item not added!';
                            }
                        } else {
                            $res['status'] = 0;
                            $res['message'] = 'Invoice not added!';
                        }
                        // $res['status'] = 1;
                        // $res['message'] = 'Invoice added successfully!';
                        // $res['data'] = $invoice->load('customer'); 
                    }
                } else {
                    $res['status'] = 0;
                    $res['message'] = $customerRes['message'];
                }
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on Frontend/InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function todaysSales(Request $request)
    {
        try {
            $records = Invoice::with(['staff', 'customer', 'quotation', 'invoice_item', 'invoice_item.gold_color'])
            ->today()->withCount(['customer as todaysNewCustomersCount'=>function($q){
                $q->whereDate('created_at', date('Y-m-d'));
            }])->sortable(['id' => 'desc']);

            $customer_ids = [];

            foreach($records->distinct('customer_id')->get() as $key => $record){
                array_push($customer_ids,$record->customer_id);
            }

            $customers_id = array_unique($customer_ids);

            $new_customers = User::whereIn('id',$customers_id)->today()->count();

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {

                    $q->where('invoice_no', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_name', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_email', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_phone', 'like', '%' . $request->query('search') . '%');

                    $q->orWhereHas('invoice_item', function ($query) use ($request) {
                        $query->where('item_code', 'like', '%' . $request->query('search') . '%');
                        $query->orWhereHas('gold_color', function ($query) use ($request) {
                            $query->where('name', 'like', '%' . $request->query('search') . '%');
                        });
                    });

                    $q->orWhereHas('staff', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });

                    $q->orWhereHas('customer', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));

            return view('frontend.invoices.index', compact('records','new_customers'));
        } catch (\Exception $err) {
            Log::error('Error in todaysSales on Front\InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printInvoice($invoice_id)
    {
        try {
            $invoiceRes = $this->InvoiceComponent->getInvoiceDetails($invoice_id);
            if($invoiceRes['status'] == 1){
                $invoice = $invoiceRes['data'];
                return view('frontend.invoices.print.print_invoice', compact('invoice'));
            }else{
                return back()->with('error', $invoiceRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printInvoice on Front\InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function _updateInvoiceNumber($string, $invoice_id)
    {
        $res = [];
        try {
            $record = Invoice::findOrFail($invoice_id);
            $record->invoice_no = getItemCode(self::ZEROES, $invoice_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Invoice Number updated successfully.";
                $res['data'] = $record->invoice_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Invoice Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateInvoiceNumber on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function invoiceDetails(Request $request, $invoice_id)
    {
        $res = [];
        try{
            $record = Invoice::with(['staff', 'transaction','customer'])->findOrFail($invoice_id);
            $res['status'] = 1;
            $res['title'] = 'Success';
            $res['message'] = 'Invoice Details found!';
            $res['data'] = $record;
        } catch (\Exception $err) {
            Log::error('Error in invoiceDetails on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        if(request()->ajax()){
            return response()->json($res);
        }else{
            return $res;
        }
    }

    public function generatePDF($invoice) 
    { 
        $res = [];
        try{
            // share data to view
            view()->share('frontend.invoices.pdf.invoice_pdf', $invoice);
            $pdf = PDF::loadView('frontend.invoices.pdf.invoice_pdf', ['invoice' => $invoice])->setOptions(['defaultFont' => 'sans-serif', 'isRemoteEnabled' => true]); 

            $fileName = 'Invoice-'.$invoice->invoice_no.'-'.time().'.pdf';
            $filePath = public_path(Invoice::PDF_PATH.'/'.$fileName);
            $headers = ['Content-Type: application/pdf'];
                
            $res['status'] = 1;
            $res['message'] = 'PDF downloaded successfully!';
            $res['data'] = $pdf->download($filePath, $fileName, $headers);
        } catch (\Exception $err) {
            Log::error('Error in generatePDF on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
            $res['data'] = null; 
        }
        return $res;
    }

    public function sendEmailPDF(Request $request, $invoice_id=null)
    {
        $res = [];
        try{
            if($request->ajax()){
                $invoice_id = $request->record_id;
            }

            $invoiceRes = $this->InvoiceComponent->getInvoiceDetails($invoice_id);
            if($invoiceRes['status'] == 1){
                $invoice = $invoiceRes['data'];
            }else{
                return back()->with('error', $invoiceRes['message']);
            }

            $invoicePDFRes = $this->PDFComponent->generateInvoicePDF($invoice);
            if($invoicePDFRes['status'] == 1){
                $invoicePDF = $invoicePDFRes['data'];
            }else{
                return back()->with('error', $invoicePDFRes['message']);
            }

            $mailData = [
                'name'    => $invoice->customer_name ?? $invoice->customer->fullname,
                'email'   => $invoice->customer_email ?? $invoice->customer->email,
                'message' => 'Attach with the invoice pdf.',
                'invoicePDF'     => $invoicePDF,
            ];

            Mail::to($mailData['email'])->send(new InvoicePDFMail($mailData)); 
      
            $res['status'] = 1;
            $res['message'] = 'Quotation and Order Voucher Mail sent successfully!';
        } catch (\Exception $err) {
            Log::error('Error in sendemailPDF on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function updatePaidAmount(UpdatePaymentAmountsRequest $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $record = Invoice::findOrFail($request->id);

            $data = [
                'paid_amount'     => (float) round((($record->paid_amount ?? 0) + $request->paid_amount), 2),
                'balance'         => (float) round(((($record->paid_amount ?? 0) + $request->paid_amount) - ($record->refunded_amount ?? 0)), 2),
                'due'             => (float) round(($record->total - ((($record->paid_amount ?? 0) + $request->paid_amount) - ($record->refunded_amount ?? 0))), 2),
                'payment_status'  => ((($record->total - $request->paid_amount) > 0) ? Invoice::IN_PROGRESS : Invoice::COMPLETED),
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->id,
                    'total_amount'    => $record->total,
                    'paid_amount'     => $request->paid_amount,
                    'balance'         => $record->balance,
                    'due'             => $record->due,
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();

                if($transaction->exists()){
                    DB::commit();
                    $res['status'] = 1;
                    $res['title'] = 'Success';
                    $res['message'] = "Paid Amount added successfully";
                    $res['data'] = $record->load(['transaction'=>function($q){
                        $q->orderBy('id', 'desc')->limit(1)->get();
                    }]);
                }else{
                    $res['status'] = 0;
                    $res['title'] = 'Error';
                    $res['message'] = "Unable to save transaction for the paid amount!";
                }
            } else {
                $res['status'] = 0;
                $res['title'] = 'Error';
                $res['message'] = "Invoice not updated!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updatePaidAmount on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function updatePayment(UpdatePaymentsRequest $request)
    {
       
        $res = [];
        DB::beginTransaction();
        try {
            $record = Invoice::findOrFail($request->invoice_id);
            $paid_amount = $request->paid_amount ? $request->paid_amount : $record->total;

            $data = [
                'paid_amount' => $paid_amount,
                'balance'     => $request->paid_amount ? $request->paid_amount : 0,
                'due'         => ($record->total - $paid_amount)
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->id,
                    'total_amount'    => $record->total,
                    'paid_amount'     => $paid_amount,
                    'balance'         => $record->balance,
                    'due'             => ($record->total - $paid_amount),
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();

                if($transaction->exists()){
                    DB::commit();
                    $res['status'] = 1;
                    $res['title'] = 'Success';
                    $res['message'] = "Paid Amount added successfully";
                }else{
                    $res['status'] = 0;
                    $res['title'] = 'Error';
                    $res['message'] = "Unable to save transaction for the paid amount!";
                }
            } else {
                $res['status'] = 0;
                $res['title'] = 'Error';
                $res['message'] = "Invoice not updated!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updatePayment on Front\InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function invoiceRefund($invoice_id)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $record = Invoice::with(['staff', 'transaction'])->findOrFail($invoice_id);

            $data = [
                'balance'         => 0,
                'due'             => $record->total,
                'is_refund'       => Invoice::_ONE,
                'refunded_amount' => $record->balance,
                'payment_status'  => Invoice::PENDING,
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->id,
                    'total_amount'    => $record->total,
                    'balance'         => 0,
                    'due'             => $record->total,
                    'is_refund'       => Invoice::_ONE,
                    'refunded_amount' => $record->refunded_amount,
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();
                if($transaction->exists()){
                    DB::commit();
                    return back()->with('success', 'Amount refunded successfully!');
                }else{
                    return back()->with('error', 'Unable to save transaction for the paid amount!');
                }
            } else {
                return back()->with('error', 'Invoice not updated!');
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in invoiceRefund on Front\InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }   
    }

}

