<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\SingleDropzoneComponent;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\Admin\WorksmithsRequest;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\RegisterMail;
use App\Models\Worksmith;
use File;
use Log;
use DB;

class WorksmithsController extends Controller
{

    function __construct()
    {
        $this->SingleDropzoneComponent = new SingleDropzoneComponent();
    }

    public function index(Request $request)
    {
        try {
            $records = Worksmith::sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('fullname', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('email', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('phone', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere(DB::raw("concat(COALESCE(`country_code`,''),' ', phone)"), 'LIKE', '%' . $request->query('search') . '%');
                });
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.worksmiths.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Worksmith();
            return view('admin.worksmiths.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(WorksmithsRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();

            if (!empty($validated['image'])) {
                $imagePath = Worksmith::IMAGE_PATH;
                $tmpDirPath = Worksmith::TEMP_PATH;
                $tablename = Worksmith::TABLENAME;
                $thumbImagePath = Worksmith::THUMB_IMAGE_PATH;
                $thumbSize = Worksmith::THUMB_SIZE;
                $this->SingleDropzoneComponent->storeImage($validated, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);
            }

            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);
            
            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;
            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];
            $validated['status'] = Worksmith::ACTIVE;
            $worksmith = Worksmith::create($validated);
            if($worksmith->exists()){
                DB::commit();
            if(auth()->user()->hasPermissionTo('worksmith-read'))
                {
                    return redirect()->route('admin.worksmiths.index')->with(['success' => 'Worksmith added successfully!']);
                } else{
                    return redirect()->route('admin.dashboard.index')->with(['success' => 'Worksmith added successfully!']);
            }
            }else{
                return back()->with('error', 'Worksmith not added!');
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function edit($id)
    {
        try {
            $record = Worksmith::findOrFail($id);
            return view('admin.worksmiths.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(WorksmithsRequest $request, $id)
    {
        try{
            $validated = $request->validated(); 
            $worksmith = Worksmith::findOrFail($id);
            $imagePath = Worksmith::IMAGE_PATH;
            $tmpDirPath = Worksmith::TEMP_PATH;
            $tablename = Worksmith::TABLENAME;
            $thumbImagePath = Worksmith::THUMB_IMAGE_PATH;
            $thumbSize = Worksmith::THUMB_SIZE;
            $validated['image'] = $this->SingleDropzoneComponent->updateImage($validated, $worksmith, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);

            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);

            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;

            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];

            $worksmith->fill($validated);
            $worksmith->save();
            if($worksmith->wasChanged()){

            if(auth()->user()->hasPermissionTo('worksmith-read'))
            {
                    return redirect()->route('admin.worksmiths.index')->with(['success' => 'Worksmith updated successfully!']);
                } else{
                    return redirect()->route('admin.dashboard.index')->with(['success' => 'Worksmith updated successfully!']);
            }    
            }else{
                return back()->with('error', 'Worksmith not updated!');
            }
        } catch (\Exception $err) {
            Log::error('Error in update on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Worksmith::findOrFail($id);
            if ($record->delete()) {
                if ($record->image) {
                    $imagePath =   Worksmith::IMAGE_PATH;
                    $thumbImagePath =  Worksmith::THUMB_IMAGE_PATH;
                    $this->SingleDropzoneComponent->deleteImage($record, $imagePath,$thumbImagePath);
                    
                }
                return back()->with(['success' => 'Worksmith deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on WorksmithsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
}
