<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\SingleDropzoneComponent;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StaffsRequest;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\RegisterMail;
use App\Models\Admin;
use App\Models\Role;
use App\Models\Type;
use File;
use Log;
use DB;


class StaffsController extends Controller
{

    function __construct(SingleDropzoneComponent $singleDropzone)
    {
        $this->singleDropzone = $singleDropzone;
    }

    public function index(Request $request, $tab = null)
    {
        try {
            $all = $this->allAdmins($request);
            $subAdmins = $this->allSubAdmins($request);
            $staffs = $this->allStaffs($request);
            return view('admin.staffs.index', compact(['all', 'subAdmins', 'staffs']));
        } catch (\Exception $err) {
            Log::error('Error in index on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function allAdmins($request)
    {
        $records = Admin::with('role')->whereIn('account_type', [Admin::SUBADMIN,Admin::STAFF])->sortable(['id' => 'desc']);
        $this->commonSearch($request, $records);
        $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
        return $records;
    }

    public function allSubAdmins($request)
    {
        $records = Admin::with('role')->where('account_type', Admin::SUBADMIN)->sortable(['id' => 'desc']);
        $this->commonSearch($request, $records);
        $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
        return $records;
    }

    public function allStaffs($request)
    {
        $records = Admin::with('role')->where('account_type', Admin::STAFF)->sortable(['id' => 'desc']);
        $this->commonSearch($request, $records);
        $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
        return $records;
    }


    public function commonSearch($request, $records)
    {
        if ($request->query('search')) {
            $records = $records->where(function ($q) use ($request) {
                $q->where('fullname', 'LIKE', '%' . $request->query('search') . '%');
                $q->orWhere('username', 'LIKE', '%' . $request->query('search') . '%');
                $q->orWhere('account_type', 'LIKE', '%' . $request->query('search') . '%');
                $q->orWhere('email', 'LIKE', '%' . $request->query('search') . '%');
                $q->orWhere('phone', 'LIKE', '%' . $request->query('search') . '%');
                $q->orWhere(DB::raw("concat(COALESCE(`country_code`,''),' ', phone)"), 'LIKE', '%' . $request->query('search') . '%');
            });
        }
        return $records;
    }

    public function create()
    {
        try {
            $record = new Admin();
            return view('admin.staffs.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(StaffsRequest $request)
    {
        try {
            $validated = $request->validated();

            if (!empty($validated['account_type']) && $validated['account_type'] == Admin::SUBADMIN) {
                $validated['role_id'] = null;
            }
            if (!empty($validated['image'])) {
                $imagePath = Admin::IMAGE_PATH;
                $tmpDirPath = Admin::TEMP_PATH;
                $tablename = Admin::TABLENAME;
                $thumbImagePath = Admin::THUMB_IMAGE_PATH;
                $thumbSize = Admin::THUMB_SIZE;
                $this->singleDropzone->storeImage($validated, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);
            }

            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);
            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;
            $validated['pass'] = $validated['password'];
            $validated['password'] = Hash::make($validated['password']);
            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];
            $validated['status'] = Admin::ACTIVE;
            $staff = Admin::create($validated);

            if ($validated['account_type'] == Admin::STAFF) {
                $role_name = Role::roleNameById($validated['role_id']);
                $staff->assignRole($role_name);
            }

            if ($staff) {
                $this->RegisterMail($validated);
            }
            if(auth()->user()->hasPermissionTo('staff-read'))
                {
                    return redirect()->route('admin.staffs.index')->with(['success' => $validated['account_type'] .' '.'added successfully!']);
                } else{
                    return redirect()->route('admin.dashboard.index')->with(['success' => $validated['account_type'] .' '.'added successfully!']);
            } 
        } catch (\Exception $err) {
            Log::error('Error in store on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function RegisterMail($validated)
    {
        try {
            $email = $validated['email'];
            $fullname = $validated['fullname'];
            $password = $validated['pass'];
            $user_data = [
                'Name' => $fullname,
                'Email' => $email,
                'Password' => $password,
                'Phone' => $validated['phone'],
            ];
            $maildata = [
                'name' => $fullname,
                'title' => "Welcome To Sancy ERP.",
                'message' => "You have registered succssefully!",
                'user_data' => $user_data,
            ];
            Mail::to($email)->send(new RegisterMail($maildata));
        } catch (\Exception $err) {
            Log::error('Error in RegisterMail on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function edit($id, $tab = null)
    {
        try {
            $record = Admin::findOrFail($id);
            return view('admin.staffs.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(StaffsRequest $request, $id)
    {
        try {
            $validated = $request->validated();

            if ($validated['account_type'] == Admin::SUBADMIN) {
                $validated['role_id'] = null;
            }

            $staff = Admin::findOrFail($id);
            $imagePath = Admin::IMAGE_PATH;
            $tmpDirPath = Admin::TEMP_PATH;
            $tablename = Admin::TABLENAME;
            $thumbImagePath = Admin::THUMB_IMAGE_PATH;
            $thumbSize = Admin::THUMB_SIZE;
            $validated['image'] = $this->singleDropzone->updateImage($validated, $staff, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);

            $validated['pass'] = $validated['password'];

            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);

            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;
            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];

            if (!empty($validated['password'])) {
                $validated['password'] = Hash::make($validated['password']);
            } else {
                unset($validated['password']);
            }

            $staff->fill($validated);
            $staff->save();

            if ($validated['account_type'] == Admin::STAFF) {
                // $role =  DB::table('model_has_roles')->where('model_id', $id)->first();
                $model_has_roles = DB::table('model_has_roles')->where('model_id',$id)->delete();
                
                $role_name = Role::roleNameById($validated['role_id']);
                $staff->assignRole($role_name);
            }
            if(auth()->user()->hasPermissionTo('staff-read'))
                {
                    return redirect()->route('admin.staffs.index', ['tab' => $request->tab])->with(['success' => $validated['account_type'].' '.'updated successfully!']);
                } else{
                    return redirect()->route('admin.dashboard.index')->with(['success' => $validated['account_type'] .' '.'updated successfully!']);
            } 
        } catch (\Exception $err) {
            Log::error('Error in update on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Admin::findOrFail($id);
            if ($record->delete()) {
                if ($record->image) {
                    $imagePath = Admin::IMAGE_PATH;
                    $thumbImagePath =  Admin::THUMB_IMAGE_PATH;
                    $this->singleDropzone->deleteImage($record, $imagePath,$thumbImagePath);
                }
                return back()->with(['success' => 'Satffs deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on StaffsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
}
