<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use phpDocumentor\Reflection\Types\Nullable;
use \Spatie\Permission\Models\Role;
use App\Models\Type;
use App\Models\RoleHasPermission;
use Log;
use DB;

class RolesController extends Controller
{

    public function index(Request $request)
    {
        try {
            $records = RoleHasPermission::with('role');
            if ($request->query('search')) {
                $records = $records->whereHas('role', function ($q) use ($request) {
                    $q->where('name', 'LIKE', '%' . $request->query('search') . '%');
                });
            }

            $records = $records->orderBy('role_id')->get();
            $roleHasPermissionArr = [];
            foreach ($records as $value){
                $roleHasPermissionArr[$value->role_id][$value->permission_id] = 1;
            }

            return view('admin.roles.index', compact(['records','roleHasPermissionArr']));
        } catch (\Exception $err) {
            Log::error('Error in index on TypesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            $record = new Role();
            return view('admin.roles.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|unique:roles|max:30',
                'guard_name' => 'nullable',
                'permission' => 'nullable'
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            $validated = $validator->validated();
            $validated['name'] = ucfirst($validated['name']);

            $roleData = [];
            $roleData['name'] = $validated['name'];
            $roleData['guard_name'] = \App\Models\Role::ADMIN;
            $role = Role::create($roleData);

            if (!empty($validated['permission'])) {
                $permissions = $validated['permission'];
                $role->syncPermissions($permissions);
            }
            if(auth()->user()->hasPermissionTo('role-read'))
            {
                return redirect()->route('admin.roles.index')->with(['success' => 'Role added successfully!']);
            } else{
                return redirect()->route('admin.dashboard.index')->with('success', 'Role added successfully!');
            }
            
        } catch (\Exception $err) {
            Log::error('Error in store on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $record = Role::findOrFail($id);
            $permission_list = DB::table('role_has_permissions')->where('role_id', $id)->get();
            return view('admin.roles.edit')->with(compact('record', 'permission_list'));
        } catch (\Exception $err) {
            Log::error('Error in edit on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:30|unique:roles,name,' . $id,
                'guard_name' => 'nullable',
                'permission' => 'nullable'
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            $validated = $validator->validated();
            $roleData = [];
            $roleData['name'] = ucfirst($validated['name']);
            $roleData['guard_name'] = \App\Models\Role::ADMIN;
            $role = Role::where('id', $id)->firstOrFail();
            $role->fill($roleData);
            $role->save();

            if (!empty($validated['permission'])) {
                DB::table('role_has_permissions')->where('role_id', $id)->delete();
                $permissions = $validated['permission'];
                $role->syncPermissions($permissions);
            }else{
                DB::table('role_has_permissions')->where('role_id', $id)->delete();
            }

            if(auth()->user()->hasPermissionTo('role-read'))
            {
                return redirect()->route('admin.roles.index')->with(['success' => 'Role updated successfully!']);
            } else{
                return redirect()->route('admin.dashboard.index')->with('success', 'Role updated successfully!');
            }
        } catch (\Exception $err) {
            Log::error('Error in update on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $record = Role::findOrFail($id);

            if ($record->delete()) {

                return back()->with(['success' => 'Role deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function changeStatus(Request $request)
    {
        try {
            $record = Type::findOrFail($request->id);
            $record->status = $request->status;
            if ($record->save()) {
                $error = 0;
                $message = 'Status changed to <strong>' . $record->status . '</strong>.';
            } else {
                $error = 1;
                $message = 'Unable to change status!';
            }
            return response()->json([
                'error' => $error,
                'message' => $message
            ]);
        } catch (\Exception $err) {
            Log::error('Error in changeStatus on RolesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

}
