<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\JobReviewCostCalComponent;
use App\Http\Controllers\Components\MultipleDropzoneComponent;
use App\Http\Controllers\Components\QuotationComponent;
use App\Http\Controllers\Components\CustomerComponent;
use App\Http\Controllers\Components\PDFComponent;
use App\Http\Requests\Admin\Quotations\QuotationsRequest;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Mail\QuotationPDFMail;
use App\Helpers\CommonHelper;
use Illuminate\Support\Arr;
use App\Models\QuotationLoosePiece;
use App\Models\QuotationItemImage;
use App\Models\QuotationItem;
use App\Models\InvoiceItem;
use App\Models\Quotation;
use App\Models\Jewellery;
use App\Models\Invoice;
use App\Models\Setting;
use App\Models\Category;
use App\Models\Admin;
use File;
use Mail;
use PDF;
use Log;
use DB;


class QuotationsController extends Controller
{
    const ZEROES = 5;
    const QUOTATION = 'Quotation';
    const INVOICE = 'Invoice';

    function __construct()
    {
        $this->MultipleDropzoneComponent = new MultipleDropzoneComponent();
        $this->JobReviewCostCalComponent = new JobReviewCostCalComponent();
        $this->QuotationComponent = new QuotationComponent();
        $this->CustomerComponent = new CustomerComponent();
        $this->PDFComponent = new PDFComponent();
    }

    public function index(Request $request)
    {
        try {
            $records = Quotation::with(['customer'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('quotation_no', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('total', 'like', '%' . $request->query('search') . '%');

                    $q->orWhere('customer_name', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_email', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_phone', 'like', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                        $q->orWhere('reference_date', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('customer', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.quotations.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Quotation();
            return view('admin.quotations.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function store(QuotationsRequest $request)
    {
        DB::beginTransaction();
        try {

            $res = [];
             if(!empty($request->quotation_item)){
                $inputData = $request->except(['qut', 'jewelleries']);
                $customer = $request->only(['customer_phone', 'customer_name', 'customer_email']);
                $customerRes = $this->CustomerComponent->addOrupdateCustomer($customer);

                if ($customerRes['status'] == 1) {
                    $phoneInputRes = $this->separateDetailsFromPhoneInput($request['customer_phone']);
                    $inputData['staff_name'] = Admin::nameById($request->staff_id);
                    $inputData['country_code'] = $phoneInputRes['country_code'];
                    $inputData['customer_phone'] = $phoneInputRes['phone'];
                    $inputData['customer_id'] = $customerRes['data'];
                    $currentTimestamp = time();
                    if (empty($invoiceData['reference_date'])) {
                        $inputData['reference_date'] = date('Y-m-d', $currentTimestamp);
                    }
                    if (empty($invoiceData['reference_time'])) {
                        $inputData['reference_time'] = date('H:i:s', $currentTimestamp);
                    }

                    $subtotal_arr = [];
                    foreach ($request->qut as $key => $qut_item) {
                        foreach($qut_item as $price_key => $subtotal ){
                            if (array_key_exists("checked", $subtotal)) {
                             array_push($subtotal_arr,$subtotal['subtotal']);
                            }
                        }
                     }
                    $subtotal =  array_sum($subtotal_arr);
                    $settings  = Setting::price_settings();
                    $inputData['sub_total'] = $subtotal;
                    $inputData['gst_percent'] = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                    $inputData['gst_amount'] = ($subtotal *  $inputData['gst_percent'])/ 100 ;
                    $inputData['total'] =  ($inputData['sub_total'] + $inputData['gst_amount']) ;

                    $quotation = new Quotation();
                    $quotation->fill($inputData);
                    $quotation->save();
                    if($quotation->exists()){
                    $updateQuotationNumberRes = $this->_updateQuotationNumber(self::QUOTATION, $quotation->id);

                    if ($updateQuotationNumberRes['status'] == 1) {
                        foreach ($request->qut as $key => $qut_item) {

                            if ($qut_item[QuotationItem::INVENTORY]) {
                                $inventData = $qut_item[QuotationItem::INVENTORY];
                                if (array_key_exists("checked", $inventData)) {
                                      $jewellery = Jewellery::getJewelleryByItemCode($inventData['item_code']);
                                    if($jewellery){
                                       $qutation_data['quotation_id'] =  $quotation->id;
                                        $qutation_data['item_code'] =  $jewellery->item_code;
                                        $qutation_data['jewellery_id'] = $jewellery->id;
                                        $qutation_data['type_name'] = QuotationItem::INVENTORY;
                                        $qutation_data['category_id'] = $jewellery->category_id;
                                        $qutation_data['gold_color_id'] = $jewellery->gold_color_id ?? null;
                                        $qutation_data['description'] = $jewellery->description ?? null;
                                        $qutation_data['price'] = $jewellery->review_cost->cost_price ?? null;
                                        $qutation_data['sub_total'] = $jewellery->review_cost->sub_total ?? null;
                                        $quotation_item = $quotation->quotation_item()->create($qutation_data);
                                    } else{
                                        $res['status'] = 0;
                                        $res['message'] = "Item code not Found!";
                                       }
                                }
                            }

                            if ($qut_item[QuotationItem::BESPOKE]) {
                                $bespoke_item = $qut_item[QuotationItem::BESPOKE];
                                if (array_key_exists("checked", $bespoke_item)) {

                                    $bespoke_item['type_name'] = QuotationItem::BESPOKE;
                                    $review_cost = $this->JobReviewCostCalComponent->calculateCost($bespoke_item);
                                    if($review_cost['status'] == 1){
                                        $bespoke_item['sub_total'] = $review_cost['data']['sub_total'];
                                    }
                                    if (!empty($bespoke_item['drawing_textarea_base64'])) {
                                        $drawingImageName = CommonHelper::uploadBase64Image($bespoke_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                        $bespoke_item['drawing_image'] = $drawingImageName;
                                    }

                                    $cat_name = Category::getCategoryNameById($bespoke_item['category_id']);
                                    if(ucfirst($cat_name) != Category::RING){
                                        $bespoke_item['backing_cost'] = 0;
                                        $bespoke_item['backing_type'] = null;
                                        $bespoke_item['backing_size'] = null;
                                        $bespoke_item['backing_quantity'] = null;
                                    }

                                    $quotation_item = $quotation->quotation_item()->create($bespoke_item);
                                      if (!empty($quotation_item)) {
                                        if (!empty($bespoke_item['images'])) {
                                            $this->addQuotationImages($quotation_item, $bespoke_item['images']);
                                        }
                                        if (!empty($bespoke_item['jewelleries'])) {
                                           $this->addQuotationLoosePieces($quotation_item, $bespoke_item['jewelleries']);
                                        }
                                    }
                                }
                            }

                            if ($qut_item[QuotationItem::REPAIR]) {
                                $repair_item = $qut_item[QuotationItem::REPAIR];
                                if (array_key_exists("checked", $repair_item)) {
                                    $repair_item['type_name'] = QuotationItem::REPAIR;
                                    $review_cost = $this->JobReviewCostCalComponent->calculateCost($repair_item);

                                    if($review_cost['status'] == 1){
                                        $repair_item['sub_total'] = $review_cost['data']['sub_total'];
                                    }

                                    if (!empty($repair_item['drawing_textarea_base64'])) {
                                        $drawingImageName = CommonHelper::uploadBase64Image($repair_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                        $repair_item['drawing_image'] = $drawingImageName;
                                    }
                                    $quotation_item = $quotation->quotation_item()->create($repair_item);

                                    if (!empty($quotation_item)) {
                                        if (!empty($repair_item['images'])) {
                                            $this->addQuotationImages($quotation_item, $repair_item['images']);
                                        }
                                        if (!empty($repair_item['jewelleries'])) {
                                            $this->addQuotationLoosePieces($quotation_item, $repair_item['jewelleries']);
                                        }
                                    }
                                }
                            }
                        }
                        DB::commit();
                        if(auth()->user()->hasPermissionTo('quotation-read'))
                        {
                            $redirctUrl = route('admin.quotations.index');
                        } else{
                            $redirctUrl =route('admin.dashboard.index') ;
                        }
                        $res['status'] = 1;
                        $res['message'] = "Quotation Added Successfully!";
                        $res['redirect_url'] = $redirctUrl;
                    } else {
                        $res['status'] = 0;
                        $res['message'] = $updateQuotationNumberRes['message'];
                    }
                    }else{
                        $res['status']  = 0;
                        $res['message'] =  "Quotation not added!";
                    }
                } else {
                    $res['status'] = 0;
                    $res['message'] = $customerRes['message'];
                }
            }else{
                $res['status']  = 0;
                $res['message'] =  "Please Select an Item";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }

        return response()->json($res);
    }

    public function addQuotationImages($quotation_item,$quot_images)
    {
        try {
            $res = [];
            $imagePath = QuotationItemImage::IMAGE_PATH;
            $tmpDirPath = QuotationItemImage::TEMP_PATH;
            $tableName = Quotation::TABLENAME;
            foreach ($quot_images as $val_img){
                $this->MultipleDropzoneComponent->storeImage($val_img, $imagePath, $tmpDirPath, $tableName);
                $bespoke_items = $quotation_item->quotation_item_image()->create([
                    'quotation_id' => $quotation_item->quotation_id,
                    'quotation_item_id' => $quotation_item->id,
                    'image' => $val_img,
                ]);
            }
            $res['status'] = 1;
            $res['message'] = "Quotation images added successfully!";
        } catch (\Exception $err) {
            Log::error('Error in addQuotationImages on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function addQuotationLoosePieces($quotation_item, $qutationLoosePieces)
    {
        try {
            $res = [];
            foreach ($qutationLoosePieces as $value) {
                if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                    $value['quotation_id'] = $quotation_item->quotation_id;
                    $value['quotation_item_id'] = $quotation_item->id;
                    $quotation_item->quotation_loose_piece()->create($value);
                }
            }
            $res['status'] = 1;
            $res['message'] = "Quotation loose pieces added successfully!";
        } catch (\Exception $err) {
            Log::error('Error in addQuotationLoosePieces on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function edit($id)
    {
        try {
            $records = Quotation::with(['quotation_item', 'quotation_item.quotation_item_image', 'quotation_item.quotation_loose_piece'])->findOrFail($id);
            // dd($records);
            return view('admin.quotations.edit', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in edit on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(QuotationsRequest $request)
    {
        DB::beginTransaction();
        try {

                $isChanged = 0;
                $res = [];
                $inputData = $request->except(['qut']);
                $customer = $request->only(['customer_phone', 'customer_name', 'customer_email']);
                $customerRes = $this->CustomerComponent->addOrupdateCustomer($customer);

                if ($customerRes['status'] == 1) {

                    $phoneInputRes = $this->separateDetailsFromPhoneInput($request['customer_phone']);
                    $inputData['staff_name'] = Admin::nameById($request->staff_id);
                    $inputData['country_code'] = $phoneInputRes['country_code'];
                    $inputData['customer_phone'] = $phoneInputRes['phone'];
                    $inputData['customer_id'] = $customerRes['data'];
                    $currentTimestamp = time();
                    if (empty($invoiceData['reference_date'])) {
                        $inputData['reference_date'] = date('Y-m-d', $currentTimestamp);
                    }
                    if (empty($invoiceData['reference_time'])) {
                        $inputData['reference_time'] = date('H:i:s', $currentTimestamp);
                    }


                     if(!empty($request->update_item)){
                        $subtotal_arr = [];
                        foreach ($request->qut as $key => $qut_item) {
                            foreach($qut_item as $price_key => $subtotal ){
                                if (array_key_exists("checked", $subtotal)) {
                                 array_push($subtotal_arr,$subtotal['subtotal']);
                                }
                            }
                         }

                        $subtotal =  array_sum($subtotal_arr);
                        $settings  = Setting::price_settings();
                        $inputData['sub_total'] = $subtotal;
                        $inputData['gst_percent'] = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                        $inputData['gst_amount'] = ($subtotal *  $inputData['gst_percent'])/ 100 ;
                        $inputData['total'] =  ($inputData['sub_total'] + $inputData['gst_amount']) ;
                     }
                    //  dd($inputData);

                    $quotation =  Quotation::findOrFail($inputData['id']);
                    $quotation->fill($inputData);
                    $quotation->save();

                    if ($quotation->exists()) {
                        if($quotation->wasChanged()){
                            $isChanged = 1;
                        }

                        if(!empty($request->update_item)){
                            $quotation_item = QuotationItem::where('quotation_id', $inputData['id'])->get();
                            foreach ($quotation_item as $key => $qut_item) {
                                 $quotation_item = QuotationItem::where('quotation_id', $qut_item->quotation_id)->delete();
                                 if($quotation_item){
                                    $quotation_image = QuotationItemImage::where('quotation_item_id', $qut_item->quotation_id)->get();
                                    if ($quotation_image) {
                                        foreach ($quotation_image as $value) {
                                            $old_image_path = public_path(QuotationItemImage::IMAGE_PATH) . '/' . $value->image;
                                            if (File::exists($old_image_path)) {
                                                    File::delete($old_image_path);
                                            }
                                        }
                                    }
                                 }

                                foreach ($request->qut as $key => $input_qut_item) {
                                    if ($input_qut_item[QuotationItem::INVENTORY]) {
                                        $inventData = $input_qut_item[QuotationItem::INVENTORY];
                                        if (!empty($inventData['checked'])) {
                                            $jewellery = Jewellery::getJewelleryByItemCode($inventData['item_code']);
                                            if($jewellery){
                                            $qutation_data['quotation_id'] =  $quotation->id;
                                                $qutation_data['item_code'] =  $jewellery->item_code;
                                                $qutation_data['jewellery_id'] = $jewellery->id;
                                                $qutation_data['type_name'] = QuotationItem::INVENTORY;
                                                $qutation_data['category_id'] = $jewellery->category_id;
                                                $qutation_data['gold_color_id'] = $jewellery->gold_color_id ?? null;
                                                $qutation_data['description'] = $jewellery->description ?? null;
                                                $qutation_data['price'] = $jewellery->review_cost->cost_price ?? null;
                                                $qutation_data['sub_total'] = $jewellery->review_cost->sub_total ?? null;
                                                $quotation_item = $quotation->quotation_item()->create($qutation_data);
                                            } else{
                                                $res['status'] = 0;
                                                $res['message'] = "Item code not Found!";
                                            }
                                        }
                                    }

                                    if ($input_qut_item[QuotationItem::BESPOKE]) {
                                        $bespoke_item = $input_qut_item[QuotationItem::BESPOKE];
                                        if (!empty($bespoke_item['checked'])) {
                                            $bespoke_item['type_name'] = QuotationItem::BESPOKE;
                                            $review_cost = $this->JobReviewCostCalComponent->calculateCost($bespoke_item);
                                            if($review_cost['status'] == 1){
                                                $bespoke_item['sub_total'] = $review_cost['data']['sub_total'];
                                            }
                                            if (!empty($bespoke_item['drawing_textarea_base64'])) {
                                                $drawingImageName = CommonHelper::uploadBase64Image($bespoke_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                                $bespoke_item['drawing_image'] = $drawingImageName;
                                            }
                                            $quotation_item = $quotation->quotation_item()->create($bespoke_item);
                                            if (!empty($quotation_item)) {
                                                if (!empty($bespoke_item['images'])) {
                                                    $this->addQuotationImages($quotation_item, $bespoke_item['images']);
                                                }
                                                if (!empty($bespoke_item['jewelleries'])) {
                                                $this->addQuotationLoosePieces($quotation_item, $bespoke_item['jewelleries']);
                                                }
                                            }
                                        }
                                    }

                                    if ($input_qut_item[QuotationItem::REPAIR]) {
                                        $repair_item = $input_qut_item[QuotationItem::REPAIR];
                                        if (!empty($repair_item['checked'])) {
                                            $repair_item['type_name'] = QuotationItem::REPAIR;
                                            $review_cost = $this->JobReviewCostCalComponent->calculateCost($repair_item);

                                            if($review_cost['status'] == 1){
                                                $repair_item['sub_total'] = $review_cost['data']['sub_total'];
                                            }

                                            if (!empty($repair_item['drawing_textarea_base64'])) {

                                                $drawingImageName = CommonHelper::uploadBase64Image($repair_item['drawing_textarea_base64'], QuotationItem::DRAWING_PATH);
                                                $repair_item['drawing_image'] = $drawingImageName;
                                            }

                                            $quotation_item = $quotation->quotation_item()->create($repair_item);

                                            if (!empty($quotation_item)) {
                                                if (!empty($repair_item['images'])) {
                                                    $this->addQuotationImages($quotation_item, $repair_item['images']);
                                                }
                                                if (!empty($repair_item['jewelleries'])) {
                                                    $this->addQuotationLoosePieces($quotation_item, $repair_item['jewelleries']);
                                                }
                                            }
                                        }
                                    }
                                }

                            }
                        }
                        DB::commit();
                        $res['status'] = 1;
                        $res['message'] = "Quotation Updated Successfully!";
                    } else {
                        $res['status'] = 0;
                        $res['message'] = "Quotation not Added!" ;
                    }
                } else {
                    $res['status'] = 0;
                    $res['message'] = $customerRes['message'];
                }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in update on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function destroy($id)
    {
        try {
            $record = Quotation::with(["quotation_item.quotation_item_image"])->findOrFail($id);

            if ($record->delete()) {
                $this->_deleteItemImagePath($record);
                return back()->with(['success' => 'Quotation deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function _deleteItemImagePath($record)
    {
        try {
            $quotation_item   =   $record->quotation_item;
            foreach ($quotation_item as $key => $qut_item) {
                if ($qut_item->quotation_item_image) {
                    foreach ($qut_item->quotation_item_image as $value) {
                        $old_image_path = public_path(QuotationItemImage::IMAGE_PATH) . '/' . $value->image;
                        if (File::exists($old_image_path)) {
                                File::delete($old_image_path);
                        }
                    }
                }
            }
        } catch (\Exception $err) {
            Log::error('Error in _deleteItemImagePath on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function printQuotation($quotation_id)
    {
        try {
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
                return view('admin.quotations.print.print_quotation', compact('quotation'));
            }else{
                return back()->with('error', $quotationRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printQuotation on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printOrderVoucher($quotation_id)
    {
        try {
            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
                return view('admin.quotations.print.print_order_voucher', compact('quotation'));
            }else{
                return back()->with('error', $quotationRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printOrderVoucher on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function convertToInvoice(Request $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $quotation_id = $request->id;
            $quotation = Quotation::with('quotation_item')->findOrFail($quotation_id);
            $quotationData = is_array($quotation) ? $quotation : $quotation->toArray();
            $quotationData['quotation_id'] = $quotation->id;

            $quotationData['paid_amount'] = 0;
            $quotationData['balance'] = 0;
            $quotationData['due'] = $quotationData['total'];

            $invoice = new Invoice();
            $invoice->fill($quotationData);
            $invoice->save();

            if ($invoice->exists()) {
                $updateInvoiceNumberRes = $this->_updateInvoiceNumber(self::INVOICE, $invoice->id);

                if($updateInvoiceNumberRes['status'] == 1){
                    $invoice_id = $invoice->id;
                    $quotationItemData['invoice_id'] = $invoice_id;

                    foreach($quotationData['quotation_item'] as $key => $quotation_item){
                        $invoice_id = $invoice->id;
                        $quotation_item['invoice_id'] = $invoice_id;
                        $invoice_item = new InvoiceItem();
                        $invoice_item->fill($quotation_item);
                        $invoice_item->save();
                    }
                }

                if ($invoice_item->exists()) {
                    $invoiceItemData = InvoiceItem::where('invoice_id',$invoice_item->invoice_id)->get();
                    deleteItemsInJewellery($invoiceItemData );

                    DB::commit();
                    $res['status'] = 1;
                    $res['message'] = "Quotation converted to invoice successfully.";
                    $res['data'] = $invoice->load('customer') ?? null;
                } else {
                    $res['status'] = 0;
                    $res['message'] = "Unable to save quotation item into invoice!";
                }
            } else {
                $res['status'] = 0;
                $res['message'] = "Unable to save quotation into invoice!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in convertToInvoice on QuotationsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
        return response()->json($res);
    }

    public function _updateQuotationNumber($string, $quotation_id)
    {
        $res = [];
        try {
            $record = Quotation::findOrFail($quotation_id);
            $record->quotation_no = getItemCode(self::ZEROES, $quotation_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Quotation Number updated successfully.";
                $res['data'] = $record->quotation_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Quotation Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateQuotationNumber on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function _updateInvoiceNumber($string, $invoice_id)
    {
        $res = [];
        try {
            $record = Invoice::findOrFail($invoice_id);
            $record->invoice_no = getItemCode(self::ZEROES, $invoice_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Invoice Number updated successfully.";
                $res['data'] = $record->invoice_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Invoice Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateInvoiceNumber on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function sendEmailPDF(Request $request, $quotation_id=null)
    {
        $res = [];
        try{
            if($request->ajax()){
                $quotation_id = $request->record_id;
            }

            $quotationRes = $this->QuotationComponent->getQuotationDetails($quotation_id);
            if($quotationRes['status'] == 1){
                $quotation = $quotationRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $quotationRes['message'];
            }

            $quotationPDFRes = $this->PDFComponent->generateQuotationPDF($quotation);
            if($quotationPDFRes['status'] == 1){
                $quotationPDF = $quotationPDFRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $quotationPDFRes['message'];
            }

            $orderVoucherPDFRes = $this->PDFComponent->generateOrderVoucherPDF($quotation);
            if($orderVoucherPDFRes['status'] == 1){
                $orderVoucherPDF = $orderVoucherPDFRes['data'];
            }else{
                $res['status'] = 0;
                $res['message'] = $orderVoucherPDFRes['message'];
            }

            $mailData = [
                'name'    => $quotation->customer_name ?? $quotation->customer->fullname,
                'email'   => $quotation->customer_email ?? $quotation->customer->email,
                'message' => 'Attach with the quotation pdf and order voucher pdf.',
                'quotationPDF'     => $quotationPDF,
                'orderVoucherPDF'  => $orderVoucherPDF,
            ];

            Mail::to($mailData['email'])->send(new QuotationPDFMail($mailData));

            $res['status'] = 1;
            $res['message'] = 'Quotation and Order Voucher Mail sent successfully!';
        } catch (\Exception $err) {
            Log::error('Error in sendemailPDF on QuotationsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }


}

