<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Material;
use App\Http\Requests\Admin\MaterialsRequest;
use Log;


class MaterialsController extends Controller
{
    public function index(Request $request)
    {
        try{
            $records = Material::sortable(['id'=>'desc']);
            
            if($request->query('search')){
                $records = $records->where(function($q) use ($request){
                    $q->where('name','LIKE','%'.$request->query('search').'%');
                    $q->orWhere('material_code','LIKE','%'.$request->query('search').'%');
                    $search_array = explode('/',$request->query('search'));
                    if(count($search_array) !== 1){
                        [$search_array[0],$search_array[1]] =  [$search_array[1],$search_array[0]];
                        $new_date = implode('/',$search_array);
                        $q->orWhere('created_at','LIKE','%'.date('Y-m-d',strtotime($new_date)).'%');
                    }
                    $q->orWhereHas('material_unit',function($q) use ($request){
                        $q->where('name','LIKE','%'.$request->query('search').'%');
                    });
                });
            }
            if(($request->query('from_date')) || ($request->query('to_date')) ){

                $from_date = date('Y-m-d',strtotime($request->query('from_date')));

                $end_date  = $request->query('to_date') ? date('Y-m-d',strtotime($request->query('to_date'))) : date('Y-m-d');
                $records = $records->whereBetween('created_at',[$from_date,$end_date]);
                  
            }
            $records = $records->paginate( ($request->query('limit')?$request->query('limit'):env('PAGINATION_LIMIT')) );
            return view('admin.materials.index', compact('records'));
        }catch(\Exception $err){
            Log::error('Error in index on MaterialsController :'.$err->getMessage());
            return back()->with('error',$err->getMessage());
        }
    }

    public function create()
    {
    	try{
            $record = new Material;
            return view('admin.materials.create', compact('record'));
    	}catch(\Exception $err){
    		Log::error('Error in create on MaterialsController :'.$err->getMessage());
    		return back()->with('error',$err->getMessage());
    	}
    }

    public function store(MaterialsRequest $request)
    {
        try{     
            $validated = $request->validated();
            $materials = Material::create($validated);
            if($materials->exists()){
                $material_id = $materials->id;
                $code = str_pad(($material_id+5000),5,0,STR_PAD_LEFT);
                $material_code = 'M'.$code;
                $material_data  = Material::where('id',$material_id)->first();
                $material_data->material_code = $material_code;
                $material_data->save();
            }
            
            return redirect()->route('admin.materials.index')->with(['success'=>'Material added successfully.']);
        }catch(\Exception $err){
            Log::error('Error in store on MaterialsController :'.$err->getMessage());
            return back()->with('error',$err->getMessage());
        }
    }

    public function edit($material_id)
    {
        try{
            $record = Material::findOrFail($material_id);
            return view('admin.materials.edit', compact('record'));
        }catch(\Exception $err){
            Log::error('Error in edit on MaterialsController :'.$err->getMessage());
            return back()->with('error',$err->getMessage());
        }
    }

    public function update(MaterialsRequest $request , $material_id)
    {
        $validated = $request->validated();
        $material = Material::findOrFail($material_id);
        $material->fill($validated);
        $material->save();
        return redirect()->route('admin.materials.index')->with(['success'=>'Material updated successfully.']);
    }

    public function destroy($material_id)
    {
        try {
            $record = Material::findOrFail($material_id);
        
            if($record->delete()){
                return back()->with(['success'=>'Material deleted successfully.']);
            }else {
                return back()->with(['error'=>'Unable to delete this record.']);
            }
        } catch(\Exception $err){
            Log::error('Error in destroy on MaterialsController :'. $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function changeStatus(Request $request)
    {
        try{
            $record = Material::findOrFail($request->id);
            $record->status = $request->status;
            if($record->save()){
                $error = 0;
                $message ='Status changed to <strong>'.$record->status.'</strong>.';
            } else {
                $error = 1;
                $message ='Unable to change status.';
            }
            return response()->json([
                'error' => $error,
                'message' => $message
            ]);
        } catch(\Exception $err){
            Log::error('Error in changeStatus on MaterialsController :'. $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
}
