<?php

namespace App\Http\Controllers\admin;

use App\Helpers\CommonHelper;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TotalJobsExport;
use App\Exports\BespokeJobsExport;
use App\Exports\RepairJobsExport;
use App\Exports\DelayedJobsExport;
use Illuminate\Support\Facades\Storage;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Admin\JewelleriesController;
use Illuminate\Http\Request;
use App\Http\Requests\Admin\JewelleriesRequest;
use App\Http\Requests\Admin\JobRequests;
use App\Http\Requests\Admin\JobExportsRequest;
use App\Mail\RegisterMail;
use App\Models\JewelleryLoosePiece;
use App\Models\JewelleryImage;
use App\Models\JobLoosePiece;
use App\Models\ReviewCost;
use App\Models\Jewellery;
use App\Models\JobImage;
use App\Models\Category;
use App\Models\JobType;
use App\Models\Admin;
use App\Models\Type;
use App\Models\User;
use App\Models\Role;
use App\Models\Job;
use File;
use Mail;
use Log;
use DB;

class JobsControllerBkp extends Controller
{

    public function index(Request $request)
    {
        try {
            $records = Job::with(['review_cost', 'loose_pieces','job_type','worksmith'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    // dd($request);
                    $q->where('item_code', 'LIKE', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('estimate_completion', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orwhereHas('job_type', function ($query) use ($request) {
                        $query->where('name', 'like', '%' . $request->query('search') . '%');
                    });

                });
            }


            if (($request->query('start_date')) || ($request->query('end_date'))) {
                $start_date = (!empty($request->start_date)) ? \Carbon\Carbon::parse($request->start_date)->toDateTimeString() : \Carbon\Carbon::now();
                $end_date = (!empty($request->end_date)) ? \Carbon\Carbon::parse($request->end_date)->toDateTimeString() : \Carbon\Carbon::now();
                $records = $records->whereBetween('created_at', [$start_date, $end_date]);
            }


            if (($request->query('price_min')) || ($request->query('price_max'))) {
                $price_min = $request->price_min;
                $price_max = $request->price_max;
                $data['price_min']= $price_min;
                $data['price_max']= $price_max;
                if ($price_min && $price_max) {
                    $records = $records->whereHas('review_cost', function ($query) use($data)
                    {
                        $query->whereBetween('total', [$data['price_min'], $data['price_max']]);
                    });
                } else {
                    // $records = ($price_min) ? $records->where('price', '>=', $price_min) : $records->where('price', '<=', $price_max);
                    $records = ($data['price_min']) ? $records->whereHas('review_cost', function ($query) use($data)
                    {
                        $query->where('total', '>=', $data['price_min']);
                    })

                        : $records->whereHas('review_cost', function ($query) use($data)
                        {
                            $query->where('price', '<=',  $data['price_max']);
                        });
                }
            }

            if ($request->query('job_type_id')) {
                $records = $records->where('job_type_id', '=', $request->job_type_id);
            }
            if ($request->query('worksmith_id')) {
                $records = $records->where('worksmith_id', '=', $request->worksmith_id);
            }

            if ($request->query('job_status')) {
                $records = $records->where('job_status', '=', $request->job_status);
            }

            $records = $records->with('type', 'category')->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));

            return view('admin.jobs.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function dashboard(Request $request)
    {
        try {
            $records = Job::with(['type', 'category', 'loose_pieces', 'customer', 'review_cost'])->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));

            $worksmithRoleId = Role::roleByName(Role::WORKSMITH);

            $worksmiths = Admin::with('worksmith_job')->withCount('worksmith_job')->where(['account_type'=>Admin::STAFF, 'role_id'=>$worksmithRoleId])->orderBy('fullname')->get();

            $recentUpdates = Job::with(['type', 'category', 'loose_pieces', 'customer', 'review_cost'])->orderBy('updated_at','DESC')->take(3)->get(); 

            $jobs = Job::with(['worksmith', 'type', 'category', 'loose_pieces', 'customer', 'review_cost'])->get(); // ->paginate(3); // ->orderBy('updated_at','DESC')
            // dd($jobs);

            $timeColorsObj = ['Not Assigned'=>'#cbd5e1', 'Not Started'=>'#3F4254', 'Crafting'=>'#3E66FB', 'Completed'=>'#3CC13B', 'Received'=>'#37b981', 'Delayed'=>'#f77171'];

            $jobsList = [];
            $pairArr = [];
            $i = 1;
            $j=0;
            foreach($jobs as $key => $value){
                $pairArr[$j]['worksmithId'] = $value->worksmith ? $value->worksmith->id === '' ? 'WorkSmith-'.$key : $value->worksmith->id : 'WorkSmith-'.$key;
                $pairArr[$j]['worksmithName'] = $value->worksmith ? $value->worksmith->fullname === '' ? 'WorkSmith-'.$key : $value->worksmith->fullname : 'WorkSmith-'.$key;
                $pairArr[$j]['image'] = $value->worksmith ? asset('public/img/admins/'.$value->worksmith->image) : asset('public/img/default-placeholder.jpg');
                $pairArr[$j]['jobDesc'] = "JO-123435-1";
                $pairArr[$j]['jobStatus'] = $value->job_status ? $value->job_status : "";
                $pairArr[$j]['startDateTime'] = $value->created_at ? ($value->created_at) : date('Y-m-d H:i:s');
                $pairArr[$j]['endDateTime'] = $value->updated_at ? ($value->updated_at) :date('Y-m-d H:i:s');
                $pairArr[$j]['color'] = $value->job_status ? $timeColorsObj[$value->job_status] ?? "#D8CCA6" : "#D8CCA6";
                $j++;
                if(count($pairArr) == 3){
                    $jobsList['set'.$i] = $pairArr;
                    $pairArr = [];
                    $i++;
                    $j=0;
                }
            }

            if(!empty($pairArr)){
                $jobsList['set'.$i] = $pairArr;
                $pairArr = [];
                $i++;
            }

           // dd($jobsList);

            return view('admin.jobs.dashboard', compact(['records', 'worksmiths', 'recentUpdates', 'jobsList'])); //, 'worksmiths'
        } catch (\Exception $err) {
            Log::error('Error in dashboard on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Job();
            return view('admin.jobs.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(JobRequests $request)  //JobRequests
    {
        DB::beginTransaction();
        try {

            $validated = $request->validated();

            $job = $this->addJob($validated);

            if ($job) {
                $job_update = Job::where('id', $job->id)->update([
                    'item_code' => "J" . sprintf("%05d", $job->id)
                ]);

                if (!empty($validated['images'])) {

                    $this->addJobImages($job, $validated['images']);

                }
                if (!empty($validated['jewelleries'])) {
                    $this->addJobLoosePieces($job, $validated['jewelleries']);
                }
                DB::commit();

                if (JobType::BESPOKE == JobType::typeById($validated['type_id'])) {
                    return redirect()->route('admin.jobs.bespokeCostSummery', $job->id)->with(['success' => 'Job added successfully!']);
                } else {
                    return redirect()->route('admin.jobs.repairCostSummery', $job->id)->with(['success' => 'Job added successfully!']);
                }

            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in create on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


    public function addJob($validated)
    {
//        dd($validated);

        $validated['customer_id'] = $this->customerAddOrUpdate($validated['customer_email'], $validated);
        // ------------ signature part starts here ---------------------
        $customerSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['customer_signature'], 'img/signature') : "";
        $saleRepresentiveSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['sales_representative_signature'], 'img/signature') : "";
        if (!empty($validated['drawing_textarea_base64'])) {

            $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
            $validated['drawing_image'] = $drawingImageName;
        }

        $validated['date'] = date('Y-m-d', strtotime($validated['date']));
        $validated['customer_signature'] = !empty($customerSignature) ? $customerSignature : "";
        $validated['sales_representative_signature'] = !empty($saleRepresentiveSignature) ? $saleRepresentiveSignature : "";

        // ------------ signature part ends here -----------------------

        $validated['date'] = date('Y-m-d', strtotime($validated['date']));
        $validated['job_status'] = Job::NOT_STARTED;
        $validated['job_type_id'] = $validated['type_id'];
        $validated['craftsmanship_cost'] = $validated['workmanship_cost'];
        $validated['polishing'] = $validated['pollising_cost'];
        $job = Job::create($validated);
        return $job;

    }

    private function customerAddOrUpdate($customer_email, $data)
    {
        $user_obj = null;
        $exist_email = User::where('email', $customer_email)->first();
        $phoneNo_with_code = explode(' ', $data['customer_phone']);
        $country_code = (count($phoneNo_with_code) > 1) ? $phoneNo_with_code[0] : null;
        $phone_no = (count($phoneNo_with_code) > 1) ? $phoneNo_with_code[1] : $phoneNo_with_code[0];
        if ($exist_email) {
            if ($exist_email->phone != $phone_no) {
                $user_obj = User::where('email', $customer_email)->first();
                $user_obj->fullname = $data['customer_name'];
                $user_obj->email = $data['customer_email'];
                $user_obj->phone = $phone_no;
                $user_obj->country_code = $country_code;
                $user_obj->save();

            } else {
                $user_obj = $exist_email;
            }
        } else {
            $user = new user();
            $user->fullname = $data['customer_name'];
            $user->email = $data['customer_email'];
            $user->password = bcrypt('admin123');
            $user->phone = $phone_no;
            $user->country_code = $country_code;
            $user->status = User::ACTIVE;
            $user->save();
            $user_obj = $user;
            $user_data = [
                'Name' => $data['customer_name'],
                'Email' => $data['customer_email'],
                'Password' => "admin123",
                'Phone' => $data['customer_phone'],
            ];
            $maildata = [
                'name' => $data['customer_name'],
                'title' => "Welcome To Sancy ERP.",
                'message' => "You have registered succssefully!",
                'user_data' => $user_data,
            ];
            Mail::to($user->email)->send(new RegisterMail($maildata));
        }
        return (isset($user_obj->id)) ? $user_obj->id : null;
    }

    public function addJobImages($job, $jobImages)
    {

        foreach ($jobImages as $value) {
            if (!File::exists(public_path(Job::IMAGE_PATH))) {
                File::makeDirectory(public_path(Job::IMAGE_PATH), 0755, true, true);
            }
            if (File::exists(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value)) {
                File::copy(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value, public_path(Job::IMAGE_PATH . '/' . $value));
//                $temp_path = public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value;
//                ($temp_path) ? File::delete($temp_path) : "";
            }
            $jobImages = $job->images()->create([
                'job_id' => $job->id,
                'image' => $value,
                'status' => JobImage::ACTIVE,
            ]);

        }
        return true;
    }

    public function addJobLoosePieces($job, $jobLoosePieces)
    {
        foreach ($jobLoosePieces as $value) {
            if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                $value['job_id'] = $job->id;
                $job->loose_pieces()->create($value);
            }
        }
        return true;
    }

    public function updateJob($validatedData, $id)
    {
        $validated = $validatedData;
        $validated['customer_id'] = $this->customerAddOrUpdate($validated['customer_email'], $validated);
        // ------------ signature part starts here ---------------------

        $customerSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['customer_signature'], 'img/signature') : " ";
        $saleRepresentiveSignature = !empty(($validated['sales_representative_signature'])) ? CommonHelper::uploadBase64Image($validated['sales_representative_signature'], 'img/signature') : " ";


        if (!empty($validated['drawing_textarea_base64'])) {
            $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
            $validated['drawing_image'] = $drawingImageName;
        }

        $validated['date'] = date('Y-m-d', strtotime($validatedData['date']));
        $validated['customer_signature'] = $customerSignature;
        $validated['sales_representative_signature'] = $saleRepresentiveSignature;
        $validated['craftsmanship_cost'] = $validated['workmanship_cost'];
        $validated['polishing'] = $validated['pollising_cost'];
        $validated['job_type_id'] = $validated['type_id'];

        $job = Job::findOrfail($id);
        $job->fill($validated);
        $job->save();
        return $job;

    }

    public function edit($id)
    {
        try {
            $record = Job::with('images', 'customer')->findOrFail($id);
            if ($record) {
                $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
                $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
                $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';

            }
            return view('admin.jobs.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(JobRequests $request, $id)  //JobRequests
    {

        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $job = $this->updateJob($validated, $id);
            if ($job) {
                if (!empty($validated['images'])) {
                    $job_image = JobImage::where('job_id', $id)->get();
                    if ($job_image) {
                        foreach ($job_image as $value) {
                            $exist_image = in_array($value->image, $validated['images']);
                            if ($exist_image != true) {
                                $old_image_path = public_path(Job::IMAGE_PATH) . '/' . $value->image;
                                if (File::exists($old_image_path)) {
                                    File::delete($old_image_path);
                                }
                            }
                        }
                    }
                    JobImage::where('job_id', $id)->delete();
                    $this->addJobImages($job, $validated['images']);
                } else {
                    JobImage::where('job_id', $id)->delete();
                }

                if (!empty($validated['jewelleries'])) {
                    JobLoosePiece::where('job_id', $job->id)->delete();
                    $this->addJobLoosePieces($job, $validated['jewelleries']);
                }

                DB::commit();
                if (JobType::BESPOKE == JobType::typeById($validated['type_id'])) {
                    return redirect()->route('admin.jobs.bespokeCostSummery', $job->id)->with(['success' => 'Job updated successfully!']);
                } else {
                    return redirect()->route('admin.jobs.repairCostSummery', $job->id)->with(['success' => 'Job updated successfully!']);
                }


            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in create on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
    //-------------------------------------Duplicate Job-------------------------------------
    public function duplicateContent(Request $request,$id)
    {
        try {

            $record = Job::where('id', $id)->with(['customer', 'images', 'loose_pieces'])->first();
            $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
            $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
            $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';

            return view('admin.jobs.duplicate', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in duplicateContent on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addDuplicateJewellery(JobRequests $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $job = $this->addJob($validated);
            if ($job) {
                if ($request->has('item_code') && !empty($request->item_code)) {
                    $job_update = Job::where('id', $job->id)->first();
                    $job_update['item_code'] = $request->item_code . '-' . $job->id;
                    $job_update->save();
                }
                if (!empty($validated['jewelleries'])) {
                    $this->addJobLoosePieces($job, $validated['jewelleries']);
                }
                if (!empty($validated['images'])) {
                    $job['item_code'] = $job_update['item_code'];
                    $this->duplicateJobImages($job, $validated['images']);
                }

                if (!empty($validated['job_id'])) {
                    $reviewCostById = ReviewCost::where('job_id',$validated['job_id'])->first();
                    if($reviewCostById) {
                        $this->duplicateReviewCost($job,$reviewCostById);
                    }
                }

                DB::commit();
                return redirect()->route('admin.jobs.index')->with(['success' => 'Job added successfully!']);
            } else {
                return back()->with(['error' => 'Job not added!']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in addDuplicateJewellery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function duplicateJobImages($job, $jobImages)
    {
        try{
            foreach ($jobImages as $value) {
                $string = explode('.', $value);
                $new_string = $string[0] . '-' . $job->item_code;
                $new_image_name = $new_string . '.' . $string[1];

                if (!File::exists(public_path(Job::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Job::IMAGE_PATH), 0755, true, true);
                }

                if (File::exists(public_path(Job::IMAGE_PATH . '/' . $value))) {
                    File::copy(public_path(Job::IMAGE_PATH . '/' . $value), public_path(Job::IMAGE_PATH . '/' . $new_image_name));
                }

                $job_image = $job->images()->create([
                    'jewellery_id' => $job->id,
                    'image' => $new_image_name,
                    'status' => JobImage::ACTIVE,
                ]);
            }

            return true;
        } catch (\Exception $err) {
            Log::error('Error in duplicateJobImages on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }


    public function destroy($id)
    {
        try {
            $record = Job::findOrFail($id);
            $job_image = JobImage::where('job_id', $id)->get();
            if ($job_image) {
                foreach ($job_image as $value) {
                    $old_image_path = public_path(Job::IMAGE_PATH) . '/' . $value->image;
                    if (File::exists($old_image_path)) {
                        File::delete($old_image_path);
                    }

                }
                JobImage::where('job_id', $id)->delete();
            }

            if ($record->delete()) {
                return back()->with(['success' => 'Job deleted successfully.']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function changeStatus(Request $request)
    {
        try {
            DB::beginTransaction();
            $record = Job::with(["loose_pieces","images"])->findOrFail($request->id);
            $record->job_status = $request->status;
            if ($record->save()) {
                if ($record->job_status == Job::RECEIVED) {
                    $jewellery = new Jewellery();
                    $jewellery->job_id = $record->id;
                    $jewellery->date = date('Y-m-d', strtotime($record->date));
                    $jewellery->estimate_completion = date('Y-m-d', strtotime($record->estimate_completion));
                    $jewellery->type_id = Type::getTypeIdByName(Type::PURCHASED) ;
                    $jewellery->description = $record->description;
                    $jewellery->category_id = $record->category_id;
                    $jewellery->gold_color_id = $record->gold_color_id;
                    $jewellery->size = $record->size;
                    $jewellery->is_sample_provided = $record->is_sample_provided;
                    $jewellery->sample_description = $record->sample_description;
                    $jewellery->gold_cost = $record->gold_cost;
                    $jewellery->gold_weight = $record->gold_weight;
                    $jewellery->workmanship_cost = $record->craftsmanship_cost;
                    $jewellery->gold_weight = $record->gold_weight;
                    $jewellery->pollising_cost = $record->polishing;
                    $jewellery->setting_cost = $record->setting_cost;
                    $jewellery->earing_pin = $record->earing_pin;
                    $jewellery->backing_cost = $record->backing_cost;
                    $jewellery->backing_type = $record->backing_type;
                    $jewellery->backing_size = $record->backing_size;
                    $jewellery->backing_quantity = $record->backing_quantity;
                    $jewellery->hook = $record->hook;
                    $jewellery->claw = $record->claw;
                    $jewellery->pearl_stud = $record->pearl_stud;
                    $jewellery->staff_id = $record->staff_id;
                    $jewellery->customer_id = $record->customer_id;
                    $jewellery->drawing_image = $record->drawing_image;
                    $jewellery->save();
                    if ($jewellery->exists()) {
                        $categories = Category::where('id', $jewellery->category_id)->pluck('name')->toArray();
                        $cat_first_string = substr($categories[0], 0, 1);
                        $jewellery_update = Jewellery::where('id', $jewellery->id)->update([
                            'item_code' => $cat_first_string . sprintf("%05d", $jewellery->id)
                        ]);

                        if (!empty($record->loose_pieces) && count($record->loose_pieces)) {
                            foreach ($record->loose_pieces as $value) {
                                $jwellery_loose = $jewellery->loose_pieces()->create([
                                    "jewellery_id" => $jewellery->id,
                                    "stock_no" => $value->stock_no,
                                    "weight" => $value->weight,
                                    "unit_price" => $value->unit_price,
                                    "price" => $value->price,
                                ]);
                            }
                        }

                        if(!empty($record->images  && count($record->images)>=0 )){
                            foreach ($record->images as $value) {
                                $jwellery_image = $jewellery->images()->create([
                                    'jewellery_id' => $jewellery->id,
                                    'image' => $value->image,
                                    'status' => JewelleryImage::ACTIVE,
                                ]);

                                if (File::exists(public_path(Job::IMAGE_PATH . '/' . $value->image))) {
                                    File::copy(public_path(Job::IMAGE_PATH . '/' . $value->image), public_path(Jewellery::IMAGE_PATH . '/' . $value->image));
                                }
                            }
                        }
                        $error = 0;
                        $message = 'Status changed to <strong>' . $record->job_status . '</strong>.';
                         $status =  $record->job_status;
                    }
                   else{
                       $error = 1;
                       $message = 'Unable to change status.';
                       $status =  '';
                   }
                    DB::commit();
                    return response()->json([
                        'error' => $error,
                        'message' => $message,
                        'status'  =>$status
                    ]);
                }

                $error = 0;
                $message = 'Status changed to <strong>' . $record->job_status . '</strong>.';
                $status =  $record->job_status;
            } else {
                $error = 1;
                $message = 'Unable to change status.';
                $status = "";
            }
            DB::commit();
            return response()->json([
                'error' => $error,
                'message' => $message,
                'status' =>$status,
            ]);
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in changeStatus on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function bespokeCostSummery($id)
    {
        try {
            $record = Job::with('loose_pieces', 'review_cost')->findOrFail($id);
            return view('admin.jobs.bespoke_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in costSummery on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    public function repairCostSummery($id)
    {
        try {
            $record = Job::with('loose_pieces', 'review_cost')->findOrFail($id);
            return view('admin.jobs.repair_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in repairCostSummery on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }


    public function updateCost(Request $request)
    {
        try {
            DB::beginTransaction();
            $updateJob = Job::where('id', $request->job_id)->firstOrFail();
            $updateJob->date = date('Y-m-d', strtotime($request->date));
            $updateJob->estimate_completion = date('Y-m-d', strtotime($request->estimate_completion));
            $updateJob->gold_cost = $request->gold_cost;
            $updateJob->craftsmanship_cost = $request->craftsmanship_cost;
            $updateJob->polishing = $request->polishing;
            $updateJob->setting_cost = $request->setting_cost;
            $updateJob->backing_cost = $request->backing_cost;

            if ($updateJob->save()) {
                if (!empty($request->jewelleries)) {
                    JobLoosePiece::where('job_id', $request->job_id)->delete();
                    $this->addJobLoosePieces($updateJob, $request->jewelleries);
                }

                $cost = ReviewCost::where('job_id', $request->job_id)->first() ?? new ReviewCost();
                $cost->job_id = $request->job_id;
                $cost->mark_up = $request->mark_up;
                $cost->sub_total = $request->sub_total;
                $cost->gst_amount = !empty($request->gst_amount) ? $request->gst_amount : 0;
                $cost->total = $request->total;
                $cost->gst_percent = (!empty($request->gst_percent)) ? $request->gst_percent : 0;
                $cost->usd_exchange_rate = (!empty($request->usd) ? $request->usd : 0);
                $cost->hkd_exchange_rate = (!empty($request->hkd) ? $request->hkd : 0);
                $cost->mark_up_usd = $request->mark_up_usd;
                $cost->mark_up_hkd = $request->mark_up_hkd;
                $cost->save();
                DB::commit();
                return redirect()->route('admin.jobs.index')->with('success', 'Job updated successfully!');
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updateCost on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    private function duplicateReviewCost($job,$reviewCostById){

        try{
            $reviewCost = new ReviewCost();
            $reviewCost->job_id = $job->id;
            $reviewCost->cost_price = $reviewCostById->cost_price;
            $reviewCost->sub_total = $reviewCostById->sub_total;
            $reviewCost->gst_percent = $reviewCostById->gst_percent;
            $reviewCost->gst_amount = $reviewCostById->gst_amount;
            $reviewCost->total = $reviewCostById->total;
            $reviewCost->mark_up = $reviewCostById->mark_up;
            $reviewCost->mark_up_usd = $reviewCostById->mark_up_usd;
            $reviewCost->mark_up_hkd = $reviewCostById->mark_up_hkd;
            $reviewCost->usd_exchange_rate = $reviewCostById->usd_exchange_rate;
            $reviewCost->hkd_exchange_rate = $reviewCostById->hkd_exchange_rate;
            $reviewCost->save();
        } catch (\Exception $err) {
            Log::error('Error in duplicateReviewCost on JewelleryController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    public function printJob($id)
    {
        try {
            $record = Job::with('images', 'category', 'customer', 'loose_pieces','gold_color')->findOrFail($id);
            return view('admin.jobs.print', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in printJob on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function assignWorksmith(Request $request)
    {
        try{

            $record = Job::findOrFail($request->id);
            $record->worksmith_id = $request->worksmith_id;
            if($record->save()){
                $error = 0;
                $message ='Assign worksmith <strong>'.$request->worksmith_name.'</strong>.';
            } else {
                $error = 1;
                $message ='Unable to Assign Worksmith.';
            }
            return response()->json([
                'error' => $error,
                'message' => $message
            ]);
        } catch(\Exception $err){
            Log::error('Error in assignWorksmith on JobsController :'. $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function exportTotal(JobExportsRequest $request){
        $res = [];
        try{
            // dd($request);
            $validated = $request->validated();
            // dd($validated);

            $filename = "total_jobs-".time().".xlsx";

            $export = Excel::store(new TotalJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if($export){
                $link = Storage::url("app/public/{$filename}"); //Downloads file in sancy_erp/storage/app/public/filename.xlsx

                $res['status'] = 1;
                $res['data'] = env('APP_URL').$link;
                $res['message'] = "All Jobs exported successfully.";
            }else{
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export aLL jobs.";
            }
        } catch(\Exception $err){
            Log::error('Error in exportTotal on JobsController :'. $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportBespoke(JobExportsRequest $request){
        try{
            // dd($request);
            $validated = $request->validated();
            // dd($validated);

            $filename = "bespoke_jobs-".time().".xlsx";

            $export = Excel::store(new BespokeJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if($export){
                $link = Storage::url("app/public/{$filename}"); //Downloads file in sancy_erp/storage/app/public/filename.xlsx

                $res['status'] = 1;
                $res['data'] = env('APP_URL').$link;
                $res['message'] = "All Jobs exported successfully.";
            }else{
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export aLL jobs.";
            }
        } catch(\Exception $err){
            Log::error('Error in exportBespoke on JobsController :'. $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportRepair(JobExportsRequest $request){
        try{
            // dd($request);
            $validated = $request->validated();
            // dd($validated);

            $filename = "repair_jobs-".time().".xlsx";

            $export = Excel::store(new RepairJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if($export){
                $link = Storage::url("app/public/{$filename}"); //Downloads file in sancy_erp/storage/app/public/filename.xlsx

                $res['status'] = 1;
                $res['data'] = env('APP_URL').$link;
                $res['message'] = "All Jobs exported successfully.";
            }else{
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export aLL jobs.";
            }
        } catch(\Exception $err){
            Log::error('Error in exportRepair on JobsController :'. $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportDelayed(JobExportsRequest $request){
        try{
            // dd($request);
            $validated = $request->validated();
            // dd($validated);

            $filename = "delayed_jobs-".time().".xlsx";

            $export = Excel::store(new DelayedJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if($export){
                $link = Storage::url("app/public/{$filename}"); //Downloads file in sancy_erp/storage/app/public/filename.xlsx

                $res['status'] = 1;
                $res['data'] = env('APP_URL').$link;
                $res['message'] = "All Jobs exported successfully.";
            }else{
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export aLL jobs.";
            }
        } catch(\Exception $err){
            Log::error('Error in exportDelayed on JobsController :'. $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }


}
