<?php

namespace App\Http\Controllers\Admin;

use Intervention\Image\Facades\Image;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\Jobs\TotalJobsExport;
use App\Exports\Jobs\BespokeJobsExport;
use App\Exports\Jobs\RepairJobsExport;
use App\Exports\Jobs\DelayedJobsExport;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Components\CustomerComponent;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Admin\JewelleriesController;
use Illuminate\Http\Request;
use App\Http\Requests\Admin\Jewelleries\JewelleriesRequest;
use App\Http\Requests\Admin\Jobs\JobRequests;
use App\Http\Requests\Admin\Jobs\JobExportsRequest;
use App\Http\Requests\Admin\Jobs\JobReviewCostsRequest;
use App\Helpers\CommonHelper;
use App\Mail\RegisterMail;
use App\Models\JewelleryLoosePiece;
use App\Models\JewelleryImage;
use App\Models\JobLoosePiece;
use App\Models\ReviewCost;
use App\Models\Worksmith;
use App\Models\Jewellery;
use App\Models\JobImage;
use App\Models\Category;
use App\Models\JobType;
use App\Models\Setting;
use App\Models\Admin;
use App\Models\Type;
use App\Models\User;
use App\Models\Role;
use App\Models\Job;
use Validator;
use File;
use Mail;
use Log;
use DB;
use Carbon;


class JobsController extends Controller
{
    const JOB = 'Job';
    const ZEROES = 5;
    const JEWELLERY = 'Jewellery';
    const TABLE = 'table';
    const CALENDAR = 'calendar';
    const TIME_COLOR_ARR = Job::JOB_STATUS_COLOR_ARR;

    public function __construct()
    {
        $this->CustomerComponent = new CustomerComponent();
    }

    public function index(Request $request, $tab = null)
    {
        try {
            ini_set('max_execution_time', '300');

            if ($tab == null) {
                $tab = 'table';
            } else {
                $tab = $tab;
            }
            $newJobsItem = [];
            $newJobsList = [];

            $records = Job::with(['type', 'category', 'review_cost', 'loose_pieces', 'job_type', 'worksmith'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('item_code', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('description', 'LIKE', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('estimate_completion', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('job_type', function ($query) use ($request) {
                        $query->where('name', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            if (($request->query('start_date')) || ($request->query('end_date'))) {

                $start_date = (!empty($request->start_date)) ? \Carbon\Carbon::parse($request->start_date)->toDateTimeString() : \Carbon\Carbon::now();
                $end_date = (!empty($request->end_date)) ? \Carbon\Carbon::parse($request->end_date)->toDateTimeString() : \Carbon\Carbon::now();
                $records = $records->whereBetween('date', [$start_date, $end_date]);
            }

            if (($request->query('price_min')) || ($request->query('price_max'))) {
                $price_min = $request->price_min;
                $price_max = $request->price_max;
                $data['price_min'] = $price_min;
                $data['price_max'] = $price_max;
                if ($price_min && $price_max) {
                    $records = $records->whereHas('review_cost', function ($query) use ($data) {
                        $query->whereBetween('total', [$data['price_min'], $data['price_max']]);
                    });
                } else {
                    if ($data['price_min']) {
                        $records  = $records->whereHas('review_cost', function ($query) use ($data) {
                            $query->where('total', '>=', $data['price_min']);
                        });
                    } else {
                        $records = $records->whereHas('review_cost', function ($query) use($data) {
                         $query->where('total', '<=',  $data['price_max']);
                        });
                    }
                }
            }

            if ($request->query('job_type_id')) {
                $records = $records->where('job_type_id', '=', $request->job_type_id);
            }
            if ($request->query('worksmith_id')) {
                $records = $records->where('worksmith_id', '=', $request->worksmith_id);
            }

            if ($request->query('job_status')) {
                $records = $records->where('job_status', '=', $request->job_status);
            }

            if ($tab == self::TABLE) {
                if($request->job_status == Job::ACTIVE){
                    $records = $records->where('job_status','!=', Job::RECEIVED)->where('job_status','!=', Job::COMPLETED);
                }

                if($request->job_status == Job::ARCHIVED){
                    $records = $records->onlyTrashed();
                }
                $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            } elseif ($tab == self::CALENDAR) {
                $chartData = $this->calendarChartData($request);
                $newJobsList = $chartData['newJobsList'];
                $newJobsItem = $chartData['newJobsItem'];
            }

            return view('admin.jobs.index', compact(['tab', 'records', 'newJobsItem', 'newJobsList']));
        } catch (\Exception $err) {
            Log::error('Error in index on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function calendarChartData($request){
        try {
            $newJobsItem = [];
            $newJobsList = [];
            $year = $request->year ?? date('Y');
            $mth   = $request->month ?? date('F');
            $month = date('m', strtotime($mth));
            $cont_days  = cal_days_in_month(CAL_GREGORIAN,$month, $year);
            $date = range(1 ,  $cont_days);
            $dmy_arr = [];
            foreach($date as $key=>$vals){
             $str =  $year  . '-' . $month . '-' .$vals;
              array_push($dmy_arr ,  $str);
            }

            $startDate = $year."-".$month."-01";
            $endDate = $year."-".$month."-".$cont_days;

            $records = Job::with(['type', 'category', 'review_cost', 'loose_pieces', 'job_type', 'worksmith'])->whereNotIn('job_status', [Job::NOT_ASSIGNED])->sortable(['id' => 'desc']);
            $sql = '(';

            foreach ($dmy_arr as $key => $date) {
                if($key == 0 ){
                    $sql .= "'$date' BETWEEN date AND estimate_completion";
                }else{
                    $sql .= " OR '$date' BETWEEN date AND estimate_completion";
                }
            }
            $sql .= ')';
            $records =  $records->whereRaw(DB::raw($sql))->has('worksmith')->get();

            foreach ($records as $key => $value) {
                 $newJobsItem[] = ucfirst($value->worksmith->fullname) . ' ('.$value->item_code .') ' . ' </br> ' .$value->date . ' to '. $value->estimate_completion. '</br>';
            }

            foreach($newJobsItem as $key=>$value){
                $val = explode('(', $value);
                $vals = explode(')', $val[1]);
                $records = Job::where('item_code',$vals[0])->with('worksmith')->first();

                $startDate = $records->date;
                $endDate = $records->estimate_completion;
                if(in_array($records->date, $dmy_arr)){
                    $startDate = $records->date;
                } else{
                    $startDate = $year  . '-' . $month . '-1';
                }

                if(in_array($records->estimate_completion, $dmy_arr)){
                    $endDate = $records->estimate_completion;
                } else{
                    $endDate = $year  . '-' . $month . '-'. $cont_days;
                }
                $newJobsList[$key]['item_code'] =    $records->item_code;
                $newJobsList[$key]['x'] =  strtotime($startDate)*1000;
                $newJobsList[$key]['x2'] =  strtotime($endDate)*1000;
                $newJobsList[$key]['y']  =  $key;
                $newJobsList[$key]['worksmithName'] = $records->worksmith ? $records->worksmith->fullname === '' ? 'WorkSmith-' . $key : ucwords($records->worksmith->fullname) : 'WorkSmith-' . $key;
            }
            $charData["newJobsList"] =  $newJobsList;
            $charData["newJobsItem"] =  $newJobsItem;

            return $charData;

        } catch (\Exception $err) {
            Log::error('Error in calendarChartData on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function dashboard(Request $request)
    {
        try {
            ini_set('max_execution_time', '300');

            $records = Job::with(['type', 'category', 'loose_pieces', 'customer', 'review_cost'])->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));

            $worksmithRoleId = Role::roleByName(Role::WORKSMITH);

            $worksmiths = Worksmith::with('worksmith_job')->has('worksmith_job')->withCount('worksmith_job')->orderBy('fullname')->get();

            $recentUpdates = Job::whereMonth('created_at', date('m'))->orderBy('updated_at', 'DESC')->take(3)->get();

            $jobs = Job::with(['worksmith', 'type', 'category', 'loose_pieces', 'customer', 'review_cost'])->whereNotIn('job_status', [Job::NOT_ASSIGNED])->has('worksmith')->get();

            /* ------------ Format for data set starts here ------------- */
            $newJobsItem = [];
            $newJobsList = [];
            $jobsList = [];
            $pairArr = [];
            $i = 1;
            $j = 0;
            foreach ($jobs as $key => $value) {
                $pairArr[$j]['worksmithId'] = $value->worksmith ? $value->worksmith->id === '' ? 'WorkSmith-' . $key : $value->worksmith->id : 'WorkSmith-' . $key;
                $pairArr[$j]['worksmithName'] = $value->worksmith ? $value->worksmith->fullname === '' ? 'WorkSmith-' . $key : ucwords($value->worksmith->fullname) : 'WorkSmith-' . $key;
                $pairArr[$j]['image'] = $value->worksmith ? $value->worksmith->thumbImageUrl : getDefaultImage();
                $pairArr[$j]['jobDesc'] = $value->item_code ?? '-'; // JO-123435-1
                $pairArr[$j]['jobStatus'] = $value->job_status ? strtoupper($value->job_status) : "";
                $pairArr[$j]['startDateTime'] = $value->created_at ? ($value->created_at) : date('Y-m-d H:i:s');
                $pairArr[$j]['endDateTime'] = $value->updated_at ? ($value->updated_at) : date('Y-m-d H:i:s');
                $pairArr[$j]['color'] = $value->job_status ? self::TIME_COLOR_ARR[$value->job_status] ?? "transparent" : "transparent"; // #D8CCA6 -> yellowish color
                $j++;
                if (count($pairArr) == 3) {
                    $jobsList['set' . $i] = $pairArr;
                    $pairArr = [];
                    $i++;
                    $j = 0;
                }
            }

            if (!empty($pairArr)) {
                $jobsList['set' . $i] = $pairArr;
                $pairArr = [];
                $i++;
            }

            /* --------------- Format for data set ends here -------------- */

            /* ------------ Calender Chart starts here ------------- */
            $chartData = $this->calendarChartData($request);
            $newJobsList = $chartData['newJobsList'];
            $newJobsItem = $chartData['newJobsItem'];

            /* -------------- Format for pagination ends here ------------ */
            return view('admin.jobs.dashboard.dashboard_index', compact(['records', 'worksmiths', 'recentUpdates', 'jobsList', 'newJobsList', 'newJobsItem']));
        } catch (\Exception $err) {
            Log::error('Error in dashboard on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Job();
            return view('admin.jobs.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(JobRequests $request)
    {
        DB::beginTransaction();
        try {
            $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);
            $validated = $request->validated();

            $job = $this->addJob($validated, $customerData);
            if ($job) {
                $updateItemCodeRes = $this->_updateItemCode(self::JOB, $job->id);
                if ($updateItemCodeRes['status'] == 1) {
                    if (!empty($validated['images'])) {
                        $this->addJobImages($job, $validated['images']);
                    }
                    if (!empty($validated['jewelleries'])) {
                        $this->addJobLoosePieces($job, $validated['jewelleries']);
                    }
                    DB::commit();
                    if (JobType::BESPOKE == JobType::typeById($validated['type_id'])) {

                        return redirect()->route('admin.jobs.bespokeCostSummery', $job->id)->with(['success' => 'Job added successfully!']);
                    } else {
                        return redirect()->route('admin.jobs.repairCostSummery', $job->id)->with(['success' => 'Job added successfully!']);
                    }
                } else {
                    return back()->with(['error' => $updateItemCodeRes['message']]);
                }
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in create on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addJob($validated, $customerData)
    {
        try{
            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customerData, self::JOB);
            if ($customerRes['status'] == 1) {
                // ------------ signature part starts here ---------------------
                $customerSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['customer_signature'], 'img/signature') : "";
                $saleRepresentiveSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['sales_representative_signature'], 'img/signature') : "";
                if (!empty($validated['drawing_textarea_base64'])) {
                    $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], Job::DRAWING_PATH);
                    $validated['drawing_image'] = $drawingImageName;
                }
                $validated['customer_signature'] = !empty($customerSignature) ? $customerSignature : "";
                $validated['sales_representative_signature'] = !empty($saleRepresentiveSignature) ? $saleRepresentiveSignature : "";
                // ------------ signature part ends here -----------------------
                $cat_name = Category::getCategoryNameById($validated['category_id']);

                if(ucfirst($cat_name) != Category::RING){
                    $validated['backing_cost'] = 0;
                    $validated['backing_type'] = null;
                    $validated['backing_size'] = null;
                    $validated['backing_quantity'] = null;
                }

                $validated['date'] = date('Y-m-d', strtotime($validated['date']));
                $validated['estimate_completion'] = date('Y-m-d', strtotime($validated['estimate_completion']));
                $validated['job_status'] = $validated['worksmith_id'] ? Job::NOT_STARTED : Job::NOT_ASSIGNED;
                $validated['job_type_id'] = $validated['type_id'];
                $validated['craftsmanship_cost'] = $validated['workmanship_cost']??0;
                $validated['polishing'] = $validated['pollising_cost']??0;
                $validated['customer_id'] = $customerRes['data'];
                $job = Job::create($validated);
                if($job->exists()){
                    $this->_addReviewCost($validated,$job);
                }
                return $job;
            } else {
                return back()->with('error', $customerRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in addJob on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addJobImages($job, $jobImages)
    {
        $thumbImagePath = Job::THUMB_IMAGE_PATH;
        $thumbSize = Job::THUMB_SIZE;
        foreach ($jobImages as $value) {
            if (!File::exists(public_path(Job::IMAGE_PATH))) {
                File::makeDirectory(public_path(Job::IMAGE_PATH), 0755, true, true);
            }
            if (File::exists(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value)) {
                File::copy(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value, public_path(Job::IMAGE_PATH . '/' . $value));
            }

            if (!File::exists(public_path(Job::THUMB_IMAGE_PATH))) {
                File::makeDirectory(public_path(Job::THUMB_IMAGE_PATH), 0755, true, true);
            }
            $image = Image::make(public_path(Job::IMAGE_PATH. '/' . $value));
            $image->resize(Job::THUMB_SIZE, null, function ($constraint) {
                $constraint->aspectRatio();
            });
            $image->save(public_path(Job::THUMB_IMAGE_PATH. '/' . $value));

            $jobImages = $job->images()->create([
                'job_id' => $job->id,
                'image' => $value,
                'status' => JobImage::ACTIVE,
            ]);
        }
        return true;
    }

    public function addJobLoosePieces($job, $jobLoosePieces)
    {
        foreach ($jobLoosePieces as $value) {
            if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                $value['job_id'] = $job->id;
                $job->loose_pieces()->create($value);
            }
        }
        return true;
    }

    public function updateJob($validatedData, $id, $customerData)
    {
        $res = [];
        try{
            $validated = $validatedData;
            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customerData, self::JOB);

            if ($customerRes['status'] == 1) {
                // ------------ signature part starts here ---------------------
                $customerSignature = !empty($validated['customer_signature']) ? CommonHelper::uploadBase64Image($validated['customer_signature'], 'img/signature') : " ";
                $saleRepresentiveSignature = !empty(($validated['sales_representative_signature'])) ? CommonHelper::uploadBase64Image($validated['sales_representative_signature'], 'img/signature') : " ";


                if (!empty($validated['drawing_textarea_base64'])) {
                    $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
                    $validated['drawing_image'] = $drawingImageName;
                }

                $cat_name = Category::getCategoryNameById($validated['category_id']);
                if(ucfirst($cat_name) != Category::RING){
                    $validated['backing_cost'] = 0;
                    $validated['backing_type'] = null;
                    $validated['backing_size'] = null;
                    $validated['backing_quantity'] = null;
                }
                $validated['date'] = date('Y-m-d', strtotime($validatedData['date']));
                $validated['estimate_completion'] = date('Y-m-d', strtotime($validated['estimate_completion']));
                isset($validated['customer_signature']) ? $validated['customer_signature'] = $customerSignature : '';
                isset($validated['sales_representative_signature']) ? $validated['sales_representative_signature'] = $saleRepresentiveSignature : '';
                isset($validated['workmanship_cost']) ? $validated['craftsmanship_cost'] = $validated['workmanship_cost'] : '';
                isset($validated['pollising_cost']) ? $validated['polishing'] = $validated['pollising_cost'] : '';
                isset($validated['type_id']) ? $validated['job_type_id'] = $validated['type_id'] : '';

                $job = Job::findOrfail($id);
                $job->fill($validated);
                $job->save();
                if($job->exists()){
                    $addReviewCostRes = $this->_addReviewCost($validated,$job);
                    if($addReviewCostRes == 1){
                        $res['status'] = 1;
                        $res['data'] = $job;
                        $res['message'] = 'Review Cost added successfully!';
                    }else{
                        $res['status'] = 0;
                        $res['data'] = $job;
                        $res['message'] = 'Review Cost not added!';
                    }
                }
            } else {
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = $customerRes['message'];
            }
        } catch (\Exception $err) {
            Log::error('Error in updateJob on JobsController :' . $err->getMessage());
            $res['status'] = 1;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    // Automatically insert review cost when creating a new job
    public function _addReviewCost($validated, $job)
    {
        try{
            $setting =  Setting::price_settings();
            $review_cost_data = [];
            $stockJewelley =   collect($validated['jewelleries']);
            $priceArr = $stockJewelley->map(function ($price){
                return  $price['price'];
            });
            $priceArr  =  $priceArr->toArray();
            $calPrice = array_sum($priceArr);
            $sub_total = ($validated['gold_cost'] ?? 0) + ($validated['setting_cost'] ?? 0) + ($validated['backing_cost'] ?? 0) + ($validated['craftsmanship_cost'] ?? 0) + ($validated['polishing'] ?? 0) + $calPrice;

            $gst_percent  =  $setting['gst_percent'];
            $gst_amount = ($sub_total * $gst_percent) /100;
            $total = $sub_total + $gst_amount;

            $review_cost_data['job_id'] = $job->id;
            $review_cost_data['sub_total'] = $sub_total;
            $review_cost_data['gst_percent'] = $gst_percent;
            $review_cost_data['gst_amount'] = $gst_amount;
            $review_cost_data['total'] = $total;

            // $review_cost = new ReviewCost();
            $review_cost = ReviewCost::firstOrNew(['job_id'=>$job->id]);
            $review_cost->fill($review_cost_data);
            if($review_cost->save()){
                return true;
            }else{
                return false;
            }
        } catch (\Exception $err) {
            Log::error('Error in _addReviewCost on JobsController :' . $err->getMessage());
            return false;
        }
    }

    public function edit($id)
    {
        try {
            $record = Job::with(['images', 'customer'])->findOrFail($id);
            if ($record) {
                $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
                $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
                $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';
            }
            return view('admin.jobs.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(JobRequests $request, $id)
    {
        DB::beginTransaction();
        $res = [];
        try {
            $job = Job::findOrfail($id);
            $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);
            $validated = $request->validated();

            if($job->job_type_id  !== $validated['type_id']){
                if(JobType::BESPOKE == JobType::typeById($validated['type_id'])){
                    // Bespoke Job
                    $validated['customer_signature'] = null;
                    $validated['sales_representative_signature'] = null;
                }else{
                    // Repair Job
                    $validated['earing_pin'] = null;
                    $validated['gold_weight'] = null;
                    $validated['gold_cost'] = null;
                    $validated['craftsmanship_cost'] = null;
                    $validated['polishing'] = null;
                    $validated['gold_color_id'] = null;
                    $validated['size'] = null;
                    $validated['gold_cost'] = null;
                    $validated['is_sample_provided'] = null;
                    $validated['sample_description'] = null;
                    $validated['setting_cost'] = null;
                    $validated['backing_cost'] = null;
                    $validated['backing_type'] = null;
                    $validated['backing_size'] = null;
                    $validated['backing_quantity'] = null;
                    $validated['claw'] = null;
                }
            }

            $jobRes = $this->updateJob($validated, $id, $customerData);
            if ($jobRes['status'] = 1) {
                /* No need to update item code as item ocde will not change with the change in category field */
                if (!empty($validated['images'])) {
                    $job_image = JobImage::where('job_id', $id)->get();
                    if ($job_image) {
                        foreach ($job_image as $value) {
                            $exist_image = in_array($value->image, $validated['images']);
                            if ($exist_image != true) {
                                $old_image_path = public_path(Job::IMAGE_PATH) . '/' . $value->image;
                                if (File::exists($old_image_path)) {
                                    File::delete($old_image_path);
                                }
                            }
                        }
                    }
                    JobImage::where('job_id', $id)->delete();
                    $this->addJobImages($jobRes['data'], $validated['images']);
                } else {
                    JobImage::where('job_id', $id)->delete();
                }

                if (!empty($validated['jewelleries'])) {
                    JobLoosePiece::where('job_id', $jobRes['data']->id)->delete();
                    $this->addJobLoosePieces($jobRes['data'], $validated['jewelleries']);
                }

                DB::commit();
                if (JobType::BESPOKE == JobType::typeById($validated['type_id'])) {
                    return redirect()->route('admin.jobs.bespokeCostSummery', $jobRes['data']->id)->with(['success' => 'Job updated successfully!']);
                } else {
                    return redirect()->route('admin.jobs.repairCostSummery', $jobRes['data']->id)->with(['success' => 'Job updated successfully!']);
                }
            }else{
                return back()->with('error', $jobRes['message']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in update on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    //------------------------- Duplicate Job ------------------------

    public function duplicateContent(Request $request, $id)
    {
        try {
            $record = Job::where('id', $id)->with(['customer', 'images', 'loose_pieces'])->first();
            $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
            $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
            $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';

            return view('admin.jobs.duplicate', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in duplicateContent on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addDuplicateJob(JobRequests $request)
    {
        DB::beginTransaction();
        try {
            $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);
            $validated = $request->validated();
            $job = $this->addJob($validated, $customerData);
            if ($job) {
                if ($request->has('item_code') && !empty($request->item_code)) {
                    $job_update = Job::where('id', $job->id)->first();
                    $job_update['item_code'] = $request->item_code . '-' . $job->id;
                    $job_update->save();
                }
                if (!empty($validated['jewelleries'])) {
                    $this->addJobLoosePieces($job, $validated['jewelleries']);
                }
                if (!empty($validated['images'])) {
                    $job['item_code'] = $job_update['item_code'];
                    $this->duplicateJobImages($job, $validated['images']);
                }

                if (!empty($validated['job_id'])) {
                    $reviewCostById = ReviewCost::where('job_id', $validated['job_id'])->first();
                    if ($reviewCostById) {
                        $this->duplicateReviewCost($job, $reviewCostById);
                    }
                }

                DB::commit();
                return redirect()->route('admin.jobs.index')->with(['success' => 'Job added successfully!']);
            } else {
                return back()->with(['error' => 'Job not added!']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in addDuplicateJob on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function duplicateJobImages($job, $jobImages)
    {
        try {
            foreach ($jobImages as $value) {
                $string = explode('.', $value);
                $new_string = $string[0] . '-' . $job->item_code;
                $new_image_name = $new_string . '.' . $string[1];

                if (!File::exists(public_path(Job::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Job::IMAGE_PATH), 0755, true, true);
                }

                if (File::exists(public_path(Job::IMAGE_PATH . '/' . $value))) {
                    File::copy(public_path(Job::IMAGE_PATH . '/' . $value), public_path(Job::IMAGE_PATH . '/' . $new_image_name));
                }

                $job_image = $job->images()->create([
                    'jewellery_id' => $job->id,
                    'image' => $new_image_name,
                    'status' => JobImage::ACTIVE,
                ]);
            }
            return true;
        } catch (\Exception $err) {
            Log::error('Error in duplicateJobImages on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Job::findOrFail($id);
            $job_image = JobImage::where('job_id', $id)->get();
            if ($job_image) {
                foreach ($job_image as $value) {
                    $old_image_path = public_path(Job::IMAGE_PATH) . '/' . $value->image;
                    if (File::exists($old_image_path)) {
                        File::delete($old_image_path);
                    }

                    $old_thumb_image_path = public_path(Job::THUMB_IMAGE_PATH) . '/' . $value->image;
                    if (File::exists($old_thumb_image_path)) {
                        File::delete($old_thumb_image_path);
                    }
                }
                JobImage::where('job_id', $id)->delete();
            }

            if ($record->delete()) {
                return back()->with(['success' => 'Job deleted successfully.']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function changeStatus(Request $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $validator = Validator::make($request->all(), [
                'id' => "required|integer|not_in:0",
                'status' => "required|string|in:".Job::NOT_ASSIGNED.",".Job::NOT_STARTED.",".Job::CRAFTING.",".Job::COMPLETED.",".Job::RECEIVED .",".Job::DELAYED
            ]);

            if ($validator->fails()) {
                $res['error'] = 1;
                $res['message'] = $validator->errors()->first();
                $res['status'] = 'Select Option';
                return response()->json($res);
            }

            $record = Job::findOrFail($request->id);
            if(!$record->worksmith_id){
                $res['error'] = 1;
                $res['message'] = 'Please assign a worksmith first!';
                $res['status'] = $record->job_status;
                $res['colorClass'] = Job::dynamicColor($record->job_status);
                return response()->json($res);
            }else{
                $previousStatus = $record->job_status;
                $newStatus = $request->status;

                $arr = [];
                if ($previousStatus == Job::NOT_ASSIGNED) {
                    $arr = [Job::NOT_STARTED, Job::CRAFTING, Job::COMPLETED, Job::DELAYED, Job::RECEIVED];
                } elseif ($previousStatus == Job::NOT_STARTED) {
                    $arr = [Job::CRAFTING, Job::COMPLETED, Job::DELAYED, Job::RECEIVED];
                } elseif ($previousStatus == Job::CRAFTING) {
                    $arr = [Job::COMPLETED, Job::DELAYED, Job::RECEIVED];
                } elseif ($previousStatus == Job::COMPLETED) {
                    $arr = [Job::COMPLETED, Job::DELAYED, Job::RECEIVED];
                } elseif ($previousStatus == Job::DELAYED) {
                    $arr = [Job::CRAFTING, Job::COMPLETED, Job::RECEIVED];
                } elseif ($previousStatus == Job::RECEIVED) {
                    $arr = [];
                }

                if (in_array($newStatus, $arr)) {
                    $record->job_status = $newStatus;
                    $record->save();
                }

                if ($record->wasChanged()) {
                    if ($record->job_status == Job::RECEIVED) {
                        $jewellery = new Jewellery();
                        $jewellery->job_id = $record->id;
                        $jewellery->date = date('Y-m-d', strtotime($record->date));
                        $jewellery->type_id = Type::getTypeIdByName(Type::PURCHASED);
                        $jewellery->description = $record->description;
                        $jewellery->category_id = $record->category_id;
                        $jewellery->gold_color_id = $record->gold_color_id;
                        $jewellery->size = $record->size;
                        $jewellery->drawing_image = $record->drawing_image;
                        $jewellery->save();
                        if ($jewellery->exists()) {
                            $categoryName = Category::getCategoryNameById($jewellery->category_id);
                            $updateItemCodeRes = $this->_updateItemCode($categoryName, $jewellery->id);
                            if ($updateItemCodeRes['status'] == 1) {
                                if (!empty($record->loose_pieces) && count($record->loose_pieces)) {
                                    foreach ($record->loose_pieces as $value) {
                                        $jwellery_loose = $jewellery->loose_pieces()->create([
                                            "jewellery_id" => $jewellery->id,
                                            "stock_no" => $value->stock_no,
                                            "weight" => $value->weight,
                                            "unit_price" => $value->unit_price,
                                            "price" => $value->price,
                                        ]);
                                    }
                                }

                                if (!empty($record->images && count($record->images) >= 0)) {
                                    foreach ($record->images as $value) {
                                        $jwellery_image = $jewellery->images()->create([
                                            'jewellery_id' => $jewellery->id,
                                            'image' => $value->image,
                                            'status' => JewelleryImage::ACTIVE,
                                        ]);

                                        if (File::exists(public_path(Job::IMAGE_PATH . '/' . $value->image))) {
                                            File::copy(public_path(Job::IMAGE_PATH . '/' . $value->image), public_path(Jewellery::IMAGE_PATH . '/' . $value->image));
                                        }
                                    }
                                }
                                $res['error'] = 0;
                                $res['message'] = 'Status changed to <strong>' . $record->job_status . '</strong>.';
                                $res['status'] = $record->job_status;
                                $res['colorClass'] = Job::dynamicColor($record->job_status);
                            } else {
                                return back()->with(['error' => $updateItemCodeRes['message']]);
                            }
                        } else {
                            $res['error'] = 1;
                            $res['message'] = 'Unable to change status.';
                            $res['status'] = $record->job_status;
                            $res['colorClass'] = Job::dynamicColor($record->job_status);
                        }
                    }

                    $res['error'] = 0;
                    $res['message'] = 'Status changed to <strong>' . $record->job_status . '</strong>.';
                    $res['status'] = $record->job_status;
                    $res['colorClass'] = Job::dynamicColor($record->job_status);
                } else {
                    $res['error'] = 1;
                    $res['message'] = 'Unable to change status.';
                    $res['status'] = $record->job_status;
                    $res['colorClass'] = Job::dynamicColor($record->job_status);
                }
                DB::commit();
                return response()->json($res);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in changeStatus on JobsController :' . $err->getMessage());

            $res['error'] = 2;
            $res['message'] = $err->getMessage();
            $res['status'] = "";
            $res['colorClass'] = '';
            return response()->json($res);
        }
    }

    public function bespokeCostSummery($id)
    {
        try {
            $record = Job::with('loose_pieces', 'review_cost')->findOrFail($id);
            return view('admin.jobs.bespoke_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in costSummery on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function repairCostSummery($id)
    {
        try {
            $record = Job::with('loose_pieces', 'review_cost')->findOrFail($id);
            return view('admin.jobs.repair_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in repairCostSummery on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function updateCost(JobReviewCostsRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $jobData  = $request->only(['date','estimate_completion','gold_cost','craftsmanship_cost', 'polishing','setting_cost','backing_cost']);
            $updateJob = Job::where('id', $request->job_id)->firstOrFail();
            $jobData['date'] = date('Y-m-d', strtotime($request->date));
            $jobData['estimate_completion'] = date('Y-m-d', strtotime($request->estimate_completion));
            $jobData['item_code'] =  $validated['item_code'];
            $updateJob->fill($jobData);
            $updateJob->save();
            if ($updateJob->exists()) {
                if (!empty($request->jewelleries)) {
                    JobLoosePiece::where('job_id', $request->job_id)->delete();
                    $this->addJobLoosePieces($updateJob, $request->jewelleries);
                }

            $review_cost_data = [];
            $calPrice = 0;
            if(isset($validated['jewelleries'])){
                $stockJewelley =   collect($validated['jewelleries']);
                $priceArr = $stockJewelley->map(function ($price){
                    return  $price['price'];
                });
                $priceArr  =  $priceArr->toArray();
                $calPrice = array_sum($priceArr);
            }

            $sub_total = ($validated['gold_cost'] ?? 0) + ($validated['setting_cost'] ?? 0) + ($validated['backing_cost'] ?? 0) + ($validated['craftsmanship_cost'] ?? 0) + ($validated['polishing'] ?? 0) + $calPrice;
            $gst_percent  = ( $validated['gst_percent'] ?? 0);
            $gst_amount = ($sub_total * $gst_percent) /100;
            $total = $sub_total + $gst_amount;

            $review_cost_data['job_id'] = $updateJob->id;
            $review_cost_data['sub_total'] = $sub_total;
            $review_cost_data['gst_percent'] = $gst_percent;
            $review_cost_data['gst_amount'] = $gst_amount;
            $review_cost_data['total'] = $total;
            $review_cost_data['mark_up'] = $validated['mark_up'];
            $review_cost_data['mark_up_usd'] = $validated['mark_up_usd'];
            $review_cost_data['mark_up_hkd'] = $validated['mark_up_hkd'];
            $review_cost_data['usd_exchange_rate'] = $validated['usd_exchange_rate'];
            $review_cost_data['hkd_exchange_rate'] = $validated['hkd_exchange_rate'];

            $review_cost = ReviewCost::firstOrNew(['job_id'=>$updateJob->id]);
            $review_cost->fill($review_cost_data);
            if($review_cost->save()){
                DB::commit();

                if(auth()->user()->hasPermissionTo('job-read'))
                {
                    return redirect()->route('admin.jobs.index')->with('success', 'Review cost added successfully!');
                } elseif(auth()->user()->hasPermissionTo('job-dashboard'))
                {
                    return redirect()->route('admin.jobs.dashboard')->with('success', 'Review cost added successfully!');
                } else{
                    return redirect()->route('admin.dashboard.index')->with('success', 'Review cost added successfully!');
                }
            }
            //     $reviewCost = ReviewCost::where('job_id', $request->job_id)->first() ?? new ReviewCost();
            //     dd($reviewCost);
            //     $reviewCostData  = $request->only(['job_id','mark_up','sub_total','gst_amount', 'total','gst_percent', 'usd_exchange_rate','hkd_exchange_rate','mark_up_usd','mark_up_hkd']);
            //   dd( $reviewCostData);
            //     $reviewCost->fill($reviewCostData);
            //     $reviewCost->save();

            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updateCost on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function duplicateReviewCost($job, $reviewCostById)
    {

        try {
            $reviewCost = new ReviewCost();
            $reviewCost->job_id = $job->id;
            $reviewCost->cost_price = $reviewCostById->cost_price;
            $reviewCost->sub_total = $reviewCostById->sub_total;
            $reviewCost->gst_percent = $reviewCostById->gst_percent;
            $reviewCost->gst_amount = $reviewCostById->gst_amount;
            $reviewCost->total = $reviewCostById->total;
            $reviewCost->mark_up = $reviewCostById->mark_up;
            $reviewCost->mark_up_usd = $reviewCostById->mark_up_usd;
            $reviewCost->mark_up_hkd = $reviewCostById->mark_up_hkd;
            $reviewCost->usd_exchange_rate = $reviewCostById->usd_exchange_rate;
            $reviewCost->hkd_exchange_rate = $reviewCostById->hkd_exchange_rate;
            $reviewCost->save();
        } catch (\Exception $err) {
            Log::error('Error in duplicateReviewCost on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printJob($id)
    {
        try {
            $record = Job::with('images', 'category', 'customer', 'loose_pieces', 'gold_color')->findOrFail($id);
            return view('admin.jobs.print.print_job', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in printJob on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function assignWorksmith(Request $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $validator = Validator::make($request->all(), [
                'id'             => "required|integer|not_in:0",
                'worksmith_id'   => "required|integer|not_in:0",
                'worksmith_name' => "required|string:max:255",
            ], [
                'worksmith_id.required'   => "Worksmith is required.",
                'worksmith_name.required' => "Worksmith is required.",
            ]);

            if ($validator->fails()) {
                $res['error'] = 1;
                $res['message'] = $validator->errors()->first();
                $res['status'] = 'Select Option';
                return response()->json($res);
            }
            $record = Job::findOrFail($request->id);
            $old_status = $record->job_status;
            $record->worksmith_id = $request->worksmith_id;
            $record->save();
            if ($record->wasChanged()) {
                $record->job_status = Job::NOT_STARTED;
                $record->save();
                if($record->wasChanged()){
                    DB::commit();
                    $res['error'] = 0;
                    $res['data'] = ['id'=>$record->id, 'new_job_status'=>$record->job_status,  'new_color_class'=>Job::JOB_STATUS_CLASS_ARR[$record->job_status], 'old_job_status'=>$old_status, 'old_color_class'=>Job::JOB_STATUS_CLASS_ARR[$old_status]];
                    $res['message'] = 'Assign worksmith <strong>' . $request->worksmith_name . '</strong>.';
                }else{
                    $res['error'] = 2;
                    $res['message'] = 'Something went wrong.'; // Unable to change Job Status to NOT STARTED
                }
            } else {
                $res['error'] = 1;
                $res['message'] = 'Unable to Assign Worksmith.';
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in assignWorksmith on JobsController :' . $err->getMessage());
            $res['error'] = 1;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportTotal(JobExportsRequest $request)
    {
        $res = [];
        try {
            $validated = $request->validated();

            $filename = "total_jobs-" . time() . ".xlsx";

            $export = Excel::store(new TotalJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if ($export) {
                $link = Storage::url("app/public/{$filename}");

                $res['status'] = 1;
                $res['data'] = env('APP_URL') . $link;
                $res['message'] = "All Jobs exported successfully.";
            } else {
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export all jobs.";
            }
        } catch (\Exception $err) {
            Log::error('Error in exportTotal on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportBespoke(JobExportsRequest $request)
    {
        try {
            $validated = $request->validated();

            $filename = "bespoke_jobs-" . time() . ".xlsx";

            $export = Excel::store(new BespokeJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if ($export) {
                $link = Storage::url("app/public/{$filename}");

                $res['status'] = 1;
                $res['data'] = env('APP_URL') . $link;
                $res['message'] = "Bespoke Jobs exported successfully.";
            } else {
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export bespoke jobs.";
            }
        } catch (\Exception $err) {
            Log::error('Error in exportBespoke on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportRepair(JobExportsRequest $request)
    {
        try {
            $validated = $request->validated();

            $filename = "repair_jobs-" . time() . ".xlsx";

            $export = Excel::store(new RepairJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if ($export) {
                $link = Storage::url("app/public/{$filename}");

                $res['status'] = 1;
                $res['data'] = env('APP_URL') . $link;
                $res['message'] = "Repair Jobs exported successfully.";
            } else {
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export repair jobs.";
            }
        } catch (\Exception $err) {
            Log::error('Error in exportRepair on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function exportDelayed(JobExportsRequest $request)
    {
        try {
            $validated = $request->validated();

            $filename = "delayed_jobs-" . time() . ".xlsx";

            $export = Excel::store(new DelayedJobsExport($validated['start_date'], $validated['end_date']), $filename, 'public');
            if ($export) {
                $link = Storage::url("app/public/{$filename}");

                $res['status'] = 1;
                $res['data'] = env('APP_URL') . $link;
                $res['message'] = "Delayed Jobs exported successfully.";
            } else {
                $res['status'] = 0;
                $res['data'] = null;
                $res['message'] = "Unable to export delayed jobs.";
            }
        } catch (\Exception $err) {
            Log::error('Error in exportDelayed on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['data'] = null;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function _updateItemCode($string, $job_id)
    {
        $res = [];
        try {
            if ($string == self::JOB) {
                $record = Job::findOrFail($job_id);
            } else {
                $record = Jewellery::findOrFail($job_id);
            }

            $record->item_code = getItemCode(self::ZEROES, $job_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Item Code updated successfully.";
                $res['data'] = $record->item_code;
            } else {
                $res['status'] = 0;
                $res['message'] = "Item Code not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateItemCode on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function restore($id)
    {
        try {
            $job = Job::withTrashed()->find($id)->restore();
            if ($job) {
                return back()->with(['success' => 'Job restore successfully!']);
            } else {
                return back()->with(['error' => 'Unable to restore this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in restore on JobsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
    }

}

