<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\CustomerComponent;
use Illuminate\Database\Eloquent\Collection;
use App\Http\Controllers\Controller;
use App\Helpers\CommonHelper;
use Illuminate\Support\Facades\Response;
use Illuminate\Http\Request;
use App\Http\Requests\Admin\Jewelleries\JewelleriesRequest;
use App\Http\Requests\Admin\Jewelleries\JewelleryReviewCostsRequest;
use App\Http\Requests\Admin\Jewelleries\JobReviewCostsRequest;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use App\Mail\RegisterMail;
use Intervention\Image\Facades\Image;
use App\Models\JewelleryLoosePiece;
use App\Models\JewelleryImage;
use App\Models\JobLoosePiece;
use App\Models\ReviewCost;
use App\Models\Jewellery;
use App\Models\JobImage;
use App\Models\Category;
use App\Models\Setting;
use App\Models\User;
use App\Models\Type;
use App\Models\Job;
use File;
use Log;
use DB;

class JewelleriesController extends Controller
{
    const JOB = 'Job';
    const ZEROES = 5;
    const JEWELLERY = 'Jewellery';

    public function __construct()
    {
        $this->CustomerComponent = new CustomerComponent();
    }


    public function indexByCategory(Request $request, $category_id, $tab = null)
    {
        try {
            $records = Jewellery::with(["category", "first_image", "review_cost", "job", "loose_pieces"])->where('category_id', $category_id)->sortable(['id' => 'desc']);
            if ($request->query('search')) {
                // $records = $records->withTrashed();
                $records = $records->where(function ($q) use ($request) {
                    $q->where('item_code', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('description', 'LIKE', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('date', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('review_cost', function ($query) use ($request) {
                        $query->where('total', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('mark_up', 'like', '%' . $request->query('search') . '%');
                    });

                    $q->orWhereHas('job', function ($query) use ($request) {
                        $query->where('item_code', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            if (($request->query('start_date')) || ($request->query('end_date'))) {
                $start_date = (!empty($request->start_date)) ? \Carbon\Carbon::parse($request->start_date)->toDateTimeString() : \Carbon\Carbon::now();
                $end_date = (!empty($request->end_date)) ? \Carbon\Carbon::parse($request->end_date)->toDateTimeString() : \Carbon\Carbon::now();
                $records = $records->whereBetween('date', [$start_date, $end_date]);
            }

            if (($request->query('total_cost_min')) || ($request->query('total_cost_max'))) {

                $data['total_cost_min'] = $request->query('total_cost_min');
                $data['total_cost_max'] = $request->query('total_cost_max');
                if ($data['total_cost_min']  && $data['total_cost_max'] ) {
                    $records = $records->whereHas('review_cost', function ($query) use ($data) {
                        $query->whereBetween('total', [$data['total_cost_min'], $data['total_cost_max']]);
                    });
                } else {
                    if ($data['total_cost_min']) {
                        $records  = $records->whereHas('review_cost', function ($query) use ($data) {
                            $query->where('total', '>=', $data['total_cost_min']);
                        });
                    } else {
                        $records = $records->whereHas('review_cost', function ($query) use($data) {
                         $query->where('total', '<=',  $data['total_cost_max']);
                        });
                    }
                }
            }

            // if (($request->query('total_cost_min')) || ($request->query('total_cost_max'))) {

            //     $total_cost_min = $request->total_cost_min;
            //     $total_cost_max = $request->total_cost_max;
            //     if ($total_cost_min && $total_cost_max) {
            //         $records = $records->whereBetween('total_cost', [$total_cost_min, $total_cost_max]);
            //     } else {
            //         $records = ($total_cost_min) ? $records->where('total_cost', '>=', $total_cost_min) : $records->where('total_cost', '<=', $total_cost_max);
            //     }
            // }

            if (($request->query('workmanship_cost_min')) || ($request->query('workmanship_cost_max'))) {
                $work_cost_min = $request->workmanship_cost_min;
                $work_cost_max = $request->workmanship_cost_max;
                if ($work_cost_min && $work_cost_max) {
                    $records = $records->whereBetween('workmanship_cost', [$work_cost_min, $work_cost_max]);
                } else {
                    $records = ($work_cost_min) ? $records->where('workmanship_cost', '>=', $work_cost_min) : $records->where('workmanship_cost', '<=', $work_cost_max);
                }
            }

            if (($request->query('gold_cost_min')) || ($request->query('gold_cost_max'))) {
                $gold_cost_min = $request->gold_cost_min;
                $gold_cost_max = $request->gold_cost_max;
                if ($work_cost_min && $work_cost_max) {
                    $records = $records->whereBetween('gold_cost', [$gold_cost_min, $gold_cost_max]);
                } else {
                    $records = ($gold_cost_min) ? $records->where('gold_cost', '>=', $gold_cost_min) : $records->where('gold_cost', '<=', $gold_cost_max);
                }
            }

            if ($request->query('gold_color_id')) {
                $records = $records->where('gold_color_id', '=', $request->gold_color_id);
            }
            if ($request->query('backing_type')) {
                $records = $records->where('backing_type', '=', $request->backing_type);
            }
            if ($request->query('claw')) {
                $records = $records->where('claw', '=', $request->claw);
            }

            if($request->tab == Jewellery::ARCHIVED){
                $records = $records->onlyTrashed();
            }
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.jewelleries.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $name = Route::currentRouteName();
            $record = new Jewellery();
            return view('admin.jewelleries.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    // For Temporary Image
    public function dropzoneStore(Request $request)
    {
        try {
            $image = $request->file('file');
            $originalName = $image->getClientOriginalName();
            $imageName = 'jewellery' . time() . rand(0, 9999) . $originalName;
            if (!File::exists(public_path(Jewellery::TEMP_PATH))) {
                File::makeDirectory(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/'), 0755, true, true);
            }
            $image = $image->move(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/'), $imageName);
            if (!empty($image)) {
                return response()->json([
                    'status' => 1,
                    'file' => $imageName,
                    'message' => "Image Upoload Successfully!"
                ]);
            } else {
                return response()->json([
                    'status' => 0,
                    'message' => "Image not Upoload!"
                ]);
            }

            return response()->json([
                'status' => 0,
                'message' => "Only one image upload!"
            ]);
        } catch (\Exception $err) {
            Log::error('Error in dropzoneStore on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(JewelleriesRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            if (Type::PURCHASED == Type::typeById($validated['type_id'])) {
                $jewellery = $this->addJewellery($validated);
                if ($jewellery) {
                    $categoryName = Category::getCategoryNameById($validated['category_id']);
                    $updateItemCodeRes = $this->_updateItemCode($categoryName, $jewellery->id);
                    if($updateItemCodeRes['status'] == 1){
                        if (!empty($validated['jewelleries'])) {
                            $this->addJewelleryLoosePieces($jewellery, $validated['jewelleries']);
                        }
                        if (!empty($validated['images'])) {
                            $this->addJewelleryImages($jewellery, $validated['images']);
                        }

                        DB::commit();
                        return redirect()->route('admin.jewelleries.purchagedCostSummery', ['category_id' => $jewellery->category_id, 'id' => $jewellery->id])->with(['success' => 'Jewellery added Successfully!']);
                    }else{
                        return back()->with(['error' => $updateItemCodeRes['message']]);
                    }
                } else {
                    return back()->with(['error' => 'Jewellery not added!']);
                }
            } else {
                $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);
                $job = $this->addJob($validated,$customerData);

                if ($job) {
                    $updateItemCodeRes = $this->_updateItemCode(self::JOB, $job->id);
                    if($updateItemCodeRes['status'] == 1){
                        if (!empty($validated['images'])) {
                            $this->addJobImages($job, $validated['images']);
                        }

                        if (!empty($validated['jewelleries'])) {
                            $this->addJobLoosePieces($job, $validated['jewelleries']);
                        }
                        DB::commit();
                        return redirect()->route('admin.jewelleries.bespokeCostSummery', $job->id)->with(['success' => 'Job added successfully!']);
                    }else{
                        return back()->with(['error' => $updateItemCodeRes['message']]);
                    }
                } else {
                    return back()->with(['error' => 'Job not added!']);
                }
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    //---------------------------------------------------------------------

    private function addJewellery($validatedData)
    {
        try{
            $validated = $validatedData;
            if (!empty($validatedData['drawing_textarea_base64'])) {
                $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
                $validated['drawing_image'] = $drawingImageName;
            }
            $validated['date'] = date('Y-m-d', strtotime($validatedData['date']));
            $jewellery = Jewellery::create($validated);
            return $jewellery;
        } catch (\Exception $err) {
            Log::error('Error in addJewellery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    //--------------------------------------------------------------------
    private function addJewelleryImages($jewellery, $jewelleryImages)
    {
        try{
            foreach ($jewelleryImages as $value) {

                if (!File::exists(public_path(Jewellery::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Jewellery::IMAGE_PATH), 0755, true, true);
                }
                if (!File::exists(public_path(Jewellery::THUMB_IMAGE_PATH))) {
                    File::makeDirectory(public_path(Jewellery::THUMB_IMAGE_PATH), 0755, true, true);
                }

                // if (File::exists(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value)) {
                //     File::copy(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value, public_path(Jewellery::IMAGE_PATH . '/' . $value));
                //     $temp_path = public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value;
                //     File::delete($temp_path);

                //     $image = Image::make(public_path(Jewellery::IMAGE_PATH.'/'.$value));
                //     $image->resize(Jewellery::THUMB_SIZE, null, function ($constraint) {
                //         $constraint->aspectRatio();
                //     });
                //     $image->save(public_path(Jewellery::THUMB_IMAGE_PATH.'/'.$value));
                // }

                if (File::exists(public_path(Jewellery::TEMP_PATH.auth()->user()->id.'/jewelleries/'.$value))) {
                    File::copy(public_path(Jewellery::TEMP_PATH.auth()->user()->id.'/jewelleries').'/'.$value, public_path(Jewellery::IMAGE_PATH.'/'.$value));

                    $image = Image::make(public_path(Jewellery::IMAGE_PATH.'/'.$value));
                    $image->resize(Jewellery::THUMB_SIZE, null, function ($constraint) {
                        $constraint->aspectRatio();
                    });
                    $image->save(public_path(Jewellery::THUMB_IMAGE_PATH.'/'.$value));

                    $temp_path = public_path(Jewellery::TEMP_PATH.auth()->user()->id.'/jewelleries').'/'.$value;
                    File::delete($temp_path);
                }

                $jwellery_image = $jewellery->images()->create([
                    'jewellery_id' => $jewellery->id,
                    'image' => $value,
                    'status' => JewelleryImage::ACTIVE,
                ]);
            }
            return true;
        } catch (\Exception $err) {
            Log::error('Error in addJewelleryImages on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    //--------------------------------------------------------------------
    private function addJewelleryLoosePieces($jewellery, $jewelleryLoosePieces)
    {
        try{
            foreach ($jewelleryLoosePieces as $value) {
                if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                    $value['jewellery_id'] = $jewellery->id;
                    $jewellery->loose_pieces()->create($value);
                }
            }
            return true;
        } catch (\Exception $err) {
            Log::error('Error in addJewelleryLoosePieces on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    //-------------------------------------------------------------------

    public function edit($category_id, $id)
    {
        try {
            $record = Jewellery::with('images', 'category', 'customer', 'loose_pieces')->findOrFail($id);
            if ($record) {
                $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
                $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
                $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';
            }
            return view('admin.jewelleries.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(JewelleriesRequest $request, $id)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            if (Type::PURCHASED === Type::typeById($validated['type_id'])) {
                $old_jewellery = Jewellery::findOrfail($id);
                $old_category_id = $old_jewellery->category_id;
                $jewellery = $this->updateJewellery($validated, $id,$old_jewellery);
                if ($jewellery) {
                    // if($validated['category_id'] != $old_category_id){
                    //     $categoryName = Category::getCategoryNameById($jewellery->category_id);
                    //     dd($validated);
                    //     // $updateItemCodeRes = $this->_updateItemCode($categoryName, $id);
                    //     if($updateItemCodeRes['status'] == 0){
                    //         return back()->with(['error' => $updateItemCodeRes['message']]);
                    //     }
                    // }

                    if (!empty($validated['jewelleries'])) {
                        JewelleryLoosePiece::where('jewellery_id', $jewellery->id)->delete();
                        $this->addJewelleryLoosePieces($jewellery, $validated['jewelleries']);
                    }

                    if (!empty($validated['images'])) {
                        $jewellery_image = JewelleryImage::where('jewellery_id', $id)->get();
                        if ($jewellery_image) {
                            foreach ($jewellery_image as $value) {
                                $exist_image = in_array($value->image, $validated['images']);
                                if ($exist_image != true) {
                                    $old_image_path = public_path(Jewellery::IMAGE_PATH) . '/' . $value->image;
                                    if (File::exists($old_image_path)) {
                                        File::delete($old_image_path);
                                    }
                                }
                            }
                        }
                        JewelleryImage::where('jewellery_id', $id)->delete();
                        $this->addJewelleryImages($jewellery, $validated['images']);
                    } else {
                        JewelleryImage::where('jewellery_id', $id)->delete();
                    }

                    DB::commit();
                    return redirect()->route('admin.jewelleries.purchagedCostSummery', ['category_id' => $request->category_id, 'id' => $jewellery->id])->with(['success' => 'Jewellery updated successfully!']);
                } else {
                    return back()->with(['error' => 'Jewellery not added!']);
                }
            }else{
                return back()->with(['error' => 'Invalid type!']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in update on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function updateJewellery($validatedData, $id,$jewellery)
    {
        try{
            $validated = $validatedData;
            if (!empty($validated['drawing_textarea_base64'])) {
                $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
                $validated['drawing_image'] = $drawingImageName;
            }
            $validated['date'] = date('Y-m-d', strtotime($validatedData['date']));
            $jewellery->fill($validated);
            $jewellery->save();
            return $jewellery;
        } catch (\Exception $err) {
            Log::error('Error in updateJewellery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    //-------------------------------------------------------------------
    public function duplicateContent($category_id, $id, Request $request)
    {
        try {
            $record = Jewellery::where('id', $id)->with(['customer', 'images', 'loose_pieces'])->first();
            $record['drawing_image'] = !empty($record->drawing_image) ? asset('public/img/drawing_images/' . $record->drawing_image) : '';
            $record['customer_signature'] = !empty($record->customer_signature) ? asset('public/img/signature/' . $record->customer_signature) : '';
            $record['sales_representative_signature'] = !empty($record->sales_representative_signature) ? asset('public/img/signature/' . $record->sales_representative_signature) : '';

            return view('admin.jewelleries.duplicate', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in duplicateContent on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addDuplicateJewellery(JewelleriesRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $jewellery = $this->addJewellery($validated);
            if ($jewellery) {
                if ($request->has('item_code') && !empty($request->item_code)) {
                    $jewellery_update = Jewellery::where('id', $jewellery->id)->first();
                    $jewellery_update['item_code'] = $request->item_code . '-' . $jewellery->id;
                    $jewellery_update->save();
                }
                if (!empty($validated['jewelleries'])) {
                    $this->addJewelleryLoosePieces($jewellery, $validated['jewelleries']);
                }
                if (!empty($validated['images'])) {
                    $jewellery['item_code'] = $jewellery_update['item_code'];
                    $this->duplicateJewelleryImages($jewellery, $validated['images']);
                }

                if (!empty($validated['jewellery_id'])) {
                    $reviewCostById = ReviewCost::where('jewellery_id', $validated['jewellery_id'])->first();
                    if ($reviewCostById) {
                        $this->duplicateReviewCost($jewellery, $reviewCostById);
                    }
                }

                DB::commit();
                return redirect()->route('admin.jewelleries.indexByCategory', $request->category_id)->with(['success' => 'Jewellery added successfully!']);
            } else {
                return back()->with(['error' => 'Jewellery not added!']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in addDuplicateJewellery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function duplicateJewelleryImages($jewellery, $jewelleryImages)
    {
        try{
            foreach ($jewelleryImages as $value) {
                $string = explode('.', $value);
                $new_string = $string[0] . '-' . $jewellery->item_code;
                $new_image_name = $new_string . '.' . $string[1];

                if (!File::exists(public_path(Jewellery::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Jewellery::IMAGE_PATH), 0755, true, true);
                }

                if (File::exists(public_path(Jewellery::IMAGE_PATH . '/' . $value))) {
                    File::copy(public_path(Jewellery::IMAGE_PATH . '/' . $value), public_path(Jewellery::IMAGE_PATH . '/' . $new_image_name));
                }

                if (File::exists(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/' . '/' . $value))) {
                    File::copy(public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/' . '/' . $value), public_path(Jewellery::IMAGE_PATH . '/' . $new_image_name));
                    $temp_path = public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/' . '/' . $value);
                    File::delete($temp_path);

                }
                $jwellery_image = $jewellery->images()->create([
                    'jewellery_id' => $jewellery->id,
                    'image' => $new_image_name,
                    'status' => JewelleryImage::ACTIVE,
                ]);
            }
            return true;
        } catch (\Exception $err) {
            Log::error('Error in duplicateJewelleryImages on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }

    }

    //-----------------------   ////JOb/////   -------------------------

    public function addJob($validated,$customerData)
    {
        try {
            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customerData, self::JOB);

            if (!empty($validated['drawing_textarea_base64'])) {
                $drawingImageName = CommonHelper::uploadBase64Image($validated['drawing_textarea_base64'], 'img/drawing_images');
                $validated['drawing_image'] = $drawingImageName;
            }

            $cat_name = Category::getCategoryNameById($validated['category_id']);
            if(ucfirst($cat_name) != Category::RING){
                $validated['backing_cost'] = 0;
                $validated['backing_type'] = null;
                $validated['backing_size'] = null;
                $validated['backing_quantity'] = null;
            }
            $validated['date'] = date('Y-m-d', strtotime($validated['date']));
            $validated['estimate_completion'] = date('Y-m-d', strtotime($validated['estimate_completion']));
            $validated['job_status'] = $validated['worksmith_id'] ? Job::NOT_STARTED : Job::NOT_ASSIGNED ;
            $validated['job_type_id'] = $validated['type_id'];
            $validated['craftsmanship_cost'] = $validated['workmanship_cost'];
            $validated['polishing'] = $validated['pollising_cost'];
            $validated['customer_id'] = $customerRes['data'];
            $job = Job::create($validated);
            if($job->exists()){
                $this->_addReviewCost($validated,$job);
            }
            return $job;
        } catch (\Exception $err) {
            Log::error('Error in addJob on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function addJobImages($job, $jobImages)
    {
        foreach ($jobImages as $value) {
            if (!File::exists(public_path(Job::IMAGE_PATH))) {
                File::makeDirectory(public_path(Job::IMAGE_PATH), 0755, true, true);
            }
            if (!File::exists(public_path(Job::THUMB_IMAGE_PATH))) {
                File::makeDirectory(public_path(Job::THUMB_IMAGE_PATH), 0755, true, true);
            }
            if (File::exists(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value)) {
                File::copy(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value, public_path(Job::IMAGE_PATH . '/' . $value));

                File::copy(public_path(Job::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value, public_path(Job::THUMB_IMAGE_PATH . '/' . $value));
                    //                $temp_path = public_path(Jewellery::TEMP_PATH . auth()->user()->id . '/jewelleries/') . '/' . $value;
                    //                ($temp_path) ? File::delete($temp_path) : "";
            }
            $jobImages = $job->images()->create([
                'job_id' => $job->id,
                'image' => $value,
                'status' => JobImage::ACTIVE,
            ]);
        }
        return true;
    }

    public function addJobLoosePieces($job, $jobLoosePieces)
    {
        foreach ($jobLoosePieces as $value) {
            if (!empty($value['stock_no']) && !empty($value['weight']) && !empty($value['unit_price']) && !empty($value['price'])) {
                $value['job_id'] = $job->id;
                $job->loose_pieces()->create($value);
            }
        }
        return true;
    }

    public function _addReviewCost($validated,$job)
    {
        try{
            $setting =  Setting::price_settings();
            $review_cost_data = [];
            $stockJewelley =   collect($validated['jewelleries']);
            $priceArr = $stockJewelley->map(function ($price){
                return  $price['price'];
            });
            $priceArr  =  $priceArr->toArray();
            $calPrice= array_sum($priceArr);
            $sub_total = $validated['gold_cost']+ $validated['setting_cost']+ $validated['backing_cost']+ $validated['craftsmanship_cost']+ $validated['polishing']+ $calPrice;
            $gst_percent  =  $setting['gst_percent'];
            $gst_amount = $sub_total * $gst_percent/100;
            $total = $sub_total+$gst_amount;
            $review_cost_data['job_id'] = $job->id;
            $review_cost_data['sub_total'] = $sub_total;
            $review_cost_data['gst_percent'] = $gst_percent;
            $review_cost_data['gst_amount'] = $gst_amount;
            $review_cost_data['total'] = $total;
            $review_cost = new ReviewCost();
            $review_cost->fill($review_cost_data);
            $review_cost->save();
            if($review_cost->exists()){
                return true;
            }
        } catch (\Exception $err) {
            Log::error('Error in _addReviewCost on JobsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Jewellery::findOrFail($id);
            $jewellery_image = JewelleryImage::where('jewellery_id', $id)->get();

            if ($jewellery_image) {

                // foreach ($jewellery_image as $value) {
                //     $old_image_path = public_path(Jewellery::IMAGE_PATH) . '/' . $value->image;
                //     if (File::exists($old_image_path)) {
                //         File::delete($old_image_path);
                //     }

                //     $old_thumb_image_path = public_path(Jewellery::THUMB_IMAGE_PATH) . '/' . $value->image;
                //     if (File::exists($old_thumb_image_path)) {
                //         File::delete($old_thumb_image_path);
                //     }
                // }

                JewelleryImage::where('jewellery_id', $id)->delete();
            }
            if ($record->delete()) {
                return back()->with(['success' => 'Jewellery deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function changeStatus(Request $request)
    {
        try {
            $record = Jewellery::findOrFail($request->id);
            $record->status = $request->status;
            if ($record->save()) {
                $error = 0;
                $message = 'Status changed to <strong>' . $record->status . '</strong>.';
            } else {
                $error = 1;
                $message = 'Unable to change status.';
            }
            return response()->json([
                'error' => $error,
                'message' => $message
            ]);
        } catch (\Exception $err) {
            Log::error('Error in changeStatus on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function purchagedCostSummery($category_id, $id)
    {
        try {
            $record = Jewellery::with('review_cost')->findOrFail($id);
            return view('admin.jewelleries.purchaged_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in purchagedCostSummery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function bespokeCostSummery($id)
    {
        try {
            $record = Job::with('loose_pieces', 'review_cost')->findOrFail($id);
            return view('admin.jewelleries.bespoke_cost_summery', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in bespokeCostSummery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function updateCost(JewelleryReviewCostsRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $updateJob = Jewellery::where('id', $request->jewellery_id)->firstOrFail();
            $updateJob->date = date('Y-m-d', strtotime($request->date));
            $updateJob->item_code = $validated['item_code'];
            $updateJob->save();
            if($updateJob->exists()){
                unset($validated['date']);
                $validated['gst_percent'] =  !empty($validated['gst_percent']) ? $validated['gst_percent'] : 0;
                $validated['gst_amount'] =  !empty($validated['gst_amount']) ? $validated['gst_amount'] : 0;
                $cost = ReviewCost::where('jewellery_id', $request->jewellery_id)->first() ?? new ReviewCost();
                $cost->fill($validated);
                $cost->save();
                if($cost->exists()){
                    DB::commit();
                    if(auth()->user()->hasPermissionTo('jewellery-read'))
                    {
                        return redirect()->route('admin.jewelleries.indexByCategory', $updateJob->category_id)->with(['success' => 'Review cost added successfully!']);
                    } else{
                        return redirect()->route('admin.dashboard.index')->with('success', 'Review cost added successfully!');
                    }
                }else{
                    return back()->with(['error' => 'Something went wrong!']);
                }
            }else{
                return back()->with(['error' => 'Something went wrong!']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updateCost on JewelleryController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function jobUpdateCost(JobReviewCostsRequest $request)
    {
        DB::beginTransaction();
        try {

            $validated = $request->validated();
            $jobData  = $request->only(['date','estimate_completion','gold_cost','craftsmanship_cost', 'polishing','setting_cost','backing_cost']);
            $updateJob = Job::where('id', $request->job_id)->firstOrFail();
            $jobData['date'] = date('Y-m-d', strtotime($request->date));
            $jobData['estimate_completion'] = date('Y-m-d', strtotime($request->estimate_completion));
            $updateJob->fill($jobData);
            $updateJob->save();
            if ($updateJob->exists()) {
                if (!empty($request->jewelleries)) {
                    JobLoosePiece::where('job_id', $request->job_id)->delete();
                    $this->addJobLoosePieces($updateJob, $request->jewelleries);
                }
                $review_cost_data = [];
                $calPrice = 0;
                if(isset($validated['jewelleries'])){
                    $stockJewelley =   collect($validated['jewelleries']);
                    $priceArr = $stockJewelley->map(function ($price){
                        return  $price['price'];
                    });
                    $priceArr  =  $priceArr->toArray();
                    $calPrice = array_sum($priceArr);
                }


                $sub_total = ($validated['gold_cost'] ?? 0) + ($validated['setting_cost'] ?? 0) + ($validated['backing_cost'] ?? 0) + ($validated['craftsmanship_cost'] ?? 0) + ($validated['polishing'] ?? 0) + $calPrice;
                $gst_percent  = ( $validated['gst_percent'] ?? 0);
                $gst_amount = ($sub_total * $gst_percent) /100;
                $total = $sub_total + $gst_amount;

                $review_cost_data['job_id'] = $updateJob->id;
                $review_cost_data['sub_total'] = $sub_total;
                $review_cost_data['gst_percent'] = $gst_percent;
                $review_cost_data['gst_amount'] = $gst_amount;
                $review_cost_data['total'] = $total;
                $review_cost_data['mark_up'] = $validated['mark_up'];
                $review_cost_data['mark_up_usd'] = $validated['mark_up_usd'];
                $review_cost_data['mark_up_hkd'] = $validated['mark_up_hkd'];
                $review_cost_data['usd_exchange_rate'] = $validated['usd_exchange_rate'];
                $review_cost_data['hkd_exchange_rate'] = $validated['hkd_exchange_rate'];


                $review_cost = ReviewCost::firstOrNew(['job_id'=>$updateJob->id]);
                $review_cost->fill($review_cost_data);
               if($review_cost->save()){
                    DB::commit();

                    if(auth()->user()->hasPermissionTo('job-read'))
                    {
                        return redirect()->route('admin.jobs.index')->with('success', 'Review cost added successfully!');
                    } elseif(auth()->user()->hasPermissionTo('job-dashboard'))
                    {
                        return redirect()->route('admin.jobs.dashboard')->with('success', 'Review cost added successfully!');
                    } else{
                        return redirect()->route('admin.dashboard.index')->with('success', 'Review cost added successfully!');
                    }
               } else{
                 return back()->with('error',"Review cost not added!");
               }

            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in jobUpdateCost on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    private function duplicateReviewCost($jewellery, $reviewCostById)
    {
        try {
            $reviewCost = new ReviewCost();
            $reviewCost->jewellery_id = $jewellery->id;
            $reviewCost->cost_price = $reviewCostById->cost_price;
            $reviewCost->sub_total = $reviewCostById->sub_total;
            $reviewCost->gst_percent = $reviewCostById->gst_percent;
            $reviewCost->gst_amount = $reviewCostById->gst_amount;
            $reviewCost->total = $reviewCostById->total;
            $reviewCost->mark_up = $reviewCostById->mark_up;
            $reviewCost->mark_up_usd = $reviewCostById->mark_up_usd;
            $reviewCost->mark_up_hkd = $reviewCostById->mark_up_hkd;
            $reviewCost->usd_exchange_rate = $reviewCostById->usd_exchange_rate;
            $reviewCost->hkd_exchange_rate = $reviewCostById->hkd_exchange_rate;
            $reviewCost->save();
            if($reviewCost->exists()){
                Log::info('Error in duplicateReviewCost on JewelleryController :' . 'Review Cost duplicated successfully.');
            }else{
                Log::error('Error in duplicateReviewCost on JewelleryController :' . 'Unable to duplicate review cost.');
            }
        } catch (\Exception $err) {
            Log::error('Error in duplicateReviewCost on JewelleryController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printJewellery($category_id, $id)
    {
        try {
            $record = Jewellery::with('images', 'category', 'customer', 'loose_pieces', 'gold_color')->findOrFail($id);
            return view('admin.jewelleries.print', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in printJewellery on JewelleriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printJewelleryTag($category_id, $id)
    {
        try {

            $jewellery =  Jewellery::with(["review_cost","gold_color"])->where('id',$id)->firstOrFail();
            $mark_up = ($jewellery->review_cost) ? $jewellery->review_cost->mark_up : 0;
            $code = ($mark_up) ? refinedCode($mark_up) : " ";
            $price = $mark_up;
            $item_code = $jewellery->item_code;
            $gold_color =  ($jewellery->gold_color) ? $jewellery->gold_color->name : " ";
            $desc_string = $jewellery->description;


        $print_string = "^XA
^MFN,N^PR2,2,2~SD30^PW785^LH0,0^RS8,,,1,,,,^^RFW,A,2,,A^FD".$item_code."
^FS^FO-500,60^FB380,1,,L,^A0N,50,40^FD".$item_code."
^FS^FO-500,135^FB380,1,,L,^A0N,45,30^FD".$code."
^FS^FO-650,50^FB380,1,,L,^A0N,60,30^FD".$price."
^FS^FO-505,210^FB380,1,,L,^A0N,30,30^FD".$gold_color."
^FS^FO505,240^FB185,3,,L,^A0N,30,30^FD".$desc_string."
^FS^PQ1,0,1,Y
^XZ";

//         $print_string = "^XA
// ^MFN,N^PR2,2,2~SD30^PW785^LH0,0^RS8,,,1,,,,^^RFW,A,2,,A^FD".$item_code."
// ^FS^FO-500,50^FB380,1,,L,^A0N,30,30^FD".$item_code."
// ^FS^FO-500,95^FB380,1,,L,^A0N,30,30^FD".$code."
// ^FS^FO-650,50^FB380,1,,L,^A0N,60,30^FD".$price."
// ^FS^FO-500,210^FB380,1,,L,^A0N,30,30^FD".$gold_color."
// ^FS^FO630,210^FB185,3,,L,^A0N,30,30^FD".$desc_string."
// ^FS^PQ1,0,1,Y
// ^XZ";

    $data['status'] = 1;
    $data['title'] = 'Tag code created';
    $data['message'] = 'Send print options to printer!';
    $data['data'] = $print_string;

    } catch (\Exception $err) {
        Log::error('Error in printJewelleryTag on JewelleriesController :' . $err->getMessage());
        $data['status'] = 0;
        $data['title'] = 'Something occur';
        $data['message'] = $err->getMessage();
        $data['data'] = $err->getMessage();
    }
    return $data;

    }

    public function _updateItemCode($string, $jewellery_id)
    {
        $res = [];
        try{
            if($string == self::JOB){
                $record = Job::findOrFail($jewellery_id);
            }else{
                $record = Jewellery::findOrFail($jewellery_id);
            }
            $record->item_code = getItemCode(self::ZEROES, $jewellery_id, $string);
            $record->save();
            if($record->wasChanged()){
                $res['status'] = 1;
                $res['message'] = "Item Code updated successfully.";
                $res['data'] = $record->item_code;
            }else{
                $res['status'] = 0;
                $res['message'] = "Item Code not updated!";
            }
        }catch(\Exception $err){
            Log::error('Error in _updateItemCode on JewelleriesController :'.$err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

}

