<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\CustomerComponent;
use App\Http\Controllers\Components\InvoiceComponent;
use App\Http\Controllers\Components\PDFComponent;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\Admin\Invoices\InvoicesRequest;
use App\Http\Requests\Admin\Invoices\UpdatePaymentAmountsRequest;
use App\Http\Requests\Admin\Invoices\UpdatePaymentsRequest;
use App\Mail\InvoicePDFMail;
use App\Models\Transaction;
use App\Models\InvoiceItem;
use App\Models\Jewellery;
use App\Models\Setting;
use App\Models\Invoice;
use App\Models\Admin;
use App\Models\Job;
use Mail;
use PDF;
use Log;
use DB;

class InvoicesController extends Controller
{
    const ZEROES = 5;
    const INVOICE = 'Invoice';

    function __construct()
    {
        $this->CustomerComponent = new CustomerComponent();
        $this->InvoiceComponent = new InvoiceComponent();
        $this->PDFComponent = new PDFComponent();
    }

    public function index(Request $request)
    {
        try {
            $records = Invoice::with(['staff', 'customer', 'quotation', 'invoice_item', 'invoice_item.gold_color'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {

                    $q->where('id', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('invoice_no', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_name', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_email', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_phone', 'like', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                        $q->orWhere('reference_date', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('invoice_item', function ($query) use ($request) {
                        $query->where('item_code', 'like', '%' . $request->query('search') . '%');
                        $query->orWhereHas('gold_color', function ($new_query) use ($request) {
                            $new_query->where('name', 'like', '%' . $request->query('search') . '%');
                        });
                    });

                    $q->orWhereHas('staff', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });

                    $q->orWhereHas('customer', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.invoices.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            return view('admin.invoices.create');
        } catch (\Exception $err) {
            Log::error('Error in create on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(InvoicesRequest $request)
    {
        DB::beginTransaction();
        try {

          if(!empty($request->invoice_item)){
            $res = [];
            $inputData = $request->except(['inv']);
            $customer = $request->only(['customer_phone', 'customer_name', 'customer_email']);
            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customer);

            if ($customerRes['status'] == 1) {
                $phoneInputRes = $this->separateDetailsFromPhoneInput($request['customer_phone']);
                $inputData['staff_name'] = Admin::nameById($request->staff_id);
                $inputData['country_code'] = $phoneInputRes['country_code'];
                $inputData['customer_phone'] =$phoneInputRes['phone'];
                $inputData['customer_id'] = $customerRes['data'];
                $inputData['layaway'] = isset($inputData['layaway']) ? $inputData['layaway'] : 0;
                $currentTimestamp = time();
                if (empty($invoiceData['reference_date'])) {
                    $inputData['reference_date'] = date('Y-m-d', $currentTimestamp);
                }
                if (empty($invoiceData['reference_time'])) {
                    $inputData['reference_time'] = date('H:i:s', $currentTimestamp);
                }

                $price_arr = [];
                foreach ($request->inv as $key => $inv_item) {
                    foreach($inv_item as $price_key => $price ){
                        if (array_key_exists("checked", $price)) {
                            array_push($price_arr,$price['price']);
                        }
                    }
                }
                $subtotal =  array_sum($price_arr);
                $settings  = Setting::price_settings();
                $inputData['sub_total'] = $subtotal;
                $inputData['gst_percent'] = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                $inputData['gst_amount'] = ($subtotal *  $inputData['gst_percent'])/ 100 ;
                $inputData['total'] =  ($inputData['sub_total'] + $inputData['gst_amount']) ;

                $inputData['paid_amount'] = 0;
                $inputData['balance'] = 0;
                $inputData['due'] = $inputData['total'];
                $inputData['payment_status'] = Invoice::PENDING;

                $invoice = new Invoice();
                $invoice->fill($inputData);
                $invoice->save();

                if ($invoice->exists()) {
                    $updateInvoiceNumberRes = $this->_updateInvoiceNumber(self::INVOICE, $invoice->id);
                    if ($updateInvoiceNumberRes['status'] == 1) {
                        $invoice_item = [];
                        foreach ($request->inv as $key => $inv_item) {
                            foreach ($inv_item as $index => $val) {
                                if ($index == InvoiceItem::INVENTORY) {
                                    if (array_key_exists("checked", $val)) {
                                        $jewellery = Jewellery::getJewelleryByItemCode($val['item_code']);

                                        if ($jewellery) {
                                            $inven_data = [];
                                            $inven_data['invoice_id'] = $invoice->id;
                                            $inven_data['item_code'] = $jewellery->item_code;
                                            $inven_data['jewellery_id'] = $jewellery->id;
                                            $inven_data['category_id'] = $jewellery->category_id;
                                            $inven_data['gold_color_id'] = $jewellery->gold_color_id ?? null;
                                            $inven_data['description'] = $jewellery->description ?? null;
                                            $inven_data['price'] = $jewellery->review_cost->cost_price ?? null;
                                            $inven_data['sub_total'] = $jewellery->review_cost->sub_total ?? null;
                                            $inventory_item = new InvoiceItem($inven_data);
                                            array_push($invoice_item, $inventory_item);
                                        } else {
                                            $res['status'] = 0;
                                            $res['message'] = 'Item_code not exist!';
                                        }
                                    }
                                }
                                if ($index == InvoiceItem::BESPOKE || $index == InvoiceItem::REPAIR) {
                                    if (array_key_exists("checked", $val)) {
                                        $job = Job::getJobByItemCode($val['item_code']);
                                        if ($job) {
                                            $bes_rep_data = [];
                                            $bes_rep_data['invoice_id'] =  $invoice->id;
                                            $bes_rep_data['item_code'] =  $job->item_code;
                                            $bes_rep_data['job_id'] =  $job->id;
                                            $bes_rep_data['category_id'] = $job->category_id;
                                            $bes_rep_data['gold_color_id'] = $job->gold_color_id ?? null;
                                            $bes_rep_data['description'] = $job->description ?? null;
                                            $bes_rep_data['price'] = $job->review_cost->sub_total ?? null;
                                            $bes_rep_data['sub_total'] =  $job->review_cost->sub_total ?? null;
                                            $bespoke_repair_item = new InvoiceItem($bes_rep_data);
                                            array_push($invoice_item, $bespoke_repair_item);
                                        } else {
                                            $res['status'] = 0;
                                            $res['message'] = 'Item_code not exist!';
                                        }
                                    }
                                }
                            }
                        }
                        $saveInvoiceItem = $invoice->invoice_item()->saveMany($invoice_item);
                        if ($saveInvoiceItem) {
                            deleteItemsInJewellery($saveInvoiceItem);
                            DB::commit();
                            // if(auth()->user()->hasPermissionTo('invoice-read'))
                            // {
                                $redirctUrl = route('admin.invoices.cost', $invoice->id);
                            // } else{
                            //     $redirctUrl =route('admin.dashboard.index') ;
                            // } 
                            $res['status'] = 1;
                            $res['message'] = 'Invoice Item added successfully!';
                            $res['redirect_url'] = $redirctUrl;
                        } else {
                            $res['status'] = 0;
                            $res['message'] = 'Invoice Item not added!';
                        }
                    } else {
                        $res['status'] = 0;
                        $res['message'] = 'Invoice not added!';
                    }
                } else{
                    $res['status'] = 0;
                    $res['message'] = 'Invoice not added!';
                }
              } else {
                $res['status'] = 0;
                $res['message'] = $customerRes['message'];
            }
          } else{
            $res['status'] = 0;
            $res['message'] = "Please select an Item";
          }  
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function printInvoice($invoice_id)
    {
        try {
            $invoiceRes = $this->InvoiceComponent->getInvoiceDetails($invoice_id);
            if($invoiceRes['status']){
                $invoice = $invoiceRes['data'];
                return view('admin.invoices.print.print_invoice', compact('invoice'));
            }else{
                return back()->with('error', $invoiceRes['message']);
            }
        } catch (\Exception $err) {
            Log::error('Error in printInvoice on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function _updateInvoiceNumber($string, $invoice_id)
    {
        $res = [];
        try {
            $record = Invoice::findOrFail($invoice_id);
            $record->invoice_no = getItemCode(self::ZEROES, $invoice_id, $string);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Invoice Number updated successfully.";
                $res['data'] = $record->invoice_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Invoice Number not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateInvoiceNumber on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function invoiceDetails(Request $request, $invoice_id)
    {
        $res = [];
        try{
            $record = Invoice::with(['staff', 'transaction'])->findOrFail($invoice_id);
            $res['status'] = 1;
            $res['title'] = 'Success';
            $res['message'] = 'Invoice Details found!';
            $res['data'] = $record;
        } catch (\Exception $err) {
            Log::error('Error in invoiceDetails on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        if(request()->ajax()){
            return response()->json($res);
        }else{
            return $res;
        }
    }

    public function updatePaidAmount(UpdatePaymentAmountsRequest $request) 
    {
        $res = [];
        DB::beginTransaction();
        try {
            $record = Invoice::findOrFail($request->id);

            $data = [
                'paid_amount'     => (float) round((($record->paid_amount ?? 0) + $request->paid_amount), 2),
                'balance'         => (float) round(((($record->paid_amount ?? 0) + $request->paid_amount) - ($record->refunded_amount ?? 0)), 2),
                'due'             => (float) round(($record->total - ((($record->paid_amount ?? 0) + $request->paid_amount) - ($record->refunded_amount ?? 0))), 2),
                'payment_status'  => ((($record->total - $request->paid_amount) > 0) ? Invoice::IN_PROGRESS : Invoice::COMPLETED),
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->id,
                    'total_amount'    => $record->total,
                    'paid_amount'     => $request->paid_amount,
                    'balance'         => $record->balance,
                    'due'             => $record->due,
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();

                if($transaction->exists()){
                    DB::commit();
                    $res['status'] = 1;
                    $res['title'] = 'Success';
                    $res['message'] = "Paid Amount added successfully";
                    $res['data'] = $record->load(['transaction'=>function($q){
                        $q->orderBy('id', 'desc')->limit(1)->get();
                    }]);
                }else{
                    $res['status'] = 0;
                    $res['title'] = 'Error';
                    $res['message'] = "Unable to save transaction for the paid amount!";
                }
            } else {
                $res['status'] = 0;
                $res['title'] = 'Error';
                $res['message'] = "Invoice not updated!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updatePaidAmount on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function invoiceRefund($invoice_id)
    {
        $res = [];
        DB::beginTransaction();
        try {
            $record = Invoice::with(['staff', 'transaction'])->findOrFail($invoice_id);

            $data = [
                'balance'         => 0,
                'due'             => $record->total,
                'is_refund'       => Invoice::_ONE,
                'refunded_amount' => $record->balance,
                'payment_status'  => Invoice::PENDING,
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->id,
                    'total_amount'    => $record->total,
                    'balance'         => 0,
                    'due'             => $record->total,
                    'is_refund'       => Invoice::_ONE,
                    'refunded_amount' => $record->refunded_amount,
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();
                if($transaction->exists()){
                    DB::commit();
                    return back()->with('success', 'Amount refunded successfully!');
                }else{
                    return back()->with('error', 'Unable to save transaction for the paid amount!');
                }
            } else {
                return back()->with('error', 'Invoice not updated!');
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in invoiceRefund on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function sendEmailPDF(Request $request, $invoice_id=null)
    {
        $res = [];
        try{
            if($request->ajax()){
                $invoice_id = $request->record_id;
            }
            $invoiceRes = $this->InvoiceComponent->getInvoiceDetails($invoice_id);
            if($invoiceRes['status'] == 1){
                $invoice = $invoiceRes['data'];
            }else{
                return back()->with('error', $invoiceRes['message']);
            }

            $invoicePDFRes = $this->PDFComponent->generateInvoicePDF($invoice);
            if($invoicePDFRes['status'] == 1){
                $invoicePDF = $invoicePDFRes['data'];
            }else{
                return back()->with('error', $invoicePDFRes['message']);
            }

            $mailData = [
                'name'    => $invoice->customer_name ?? $invoice->customer->fullname,
                'email'   => $invoice->customer_email ?? $invoice->customer->email,
                'message' => 'Attach with the invoice pdf.',
                'invoicePDF'     => $invoicePDF,
            ];

            Mail::to($mailData['email'])->send(new InvoicePDFMail($mailData)); 

            $res['status'] = 1;
            $res['message'] = 'Quotation and Order Voucher Mail sent successfully!';
        } catch (\Exception $err) {
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

    public function updatePayment(UpdatePaymentsRequest $request)
    {
        $res = [];
        DB::beginTransaction();
        try {
           
            $record = Invoice::findOrFail($request->invoice_id);
            $paid_amount = $request->paid_amount ? $request->paid_amount : $record->total;
            
            $data = [
                'paid_amount' => $paid_amount,
                'balance'     => $request->paid_amount ? $request->paid_amount : 0,
                'due'         => ($record->total - $paid_amount)
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {
                $transaction = new Transaction();
                $transactionData = [
                    'invoice_id'      => $record->invoice_id,
                    'total_amount'    => $record->total,
                    'paid_amount'     => $paid_amount,
                    'balance'         => $record->balance,
                    'due'             => ($record->total - $paid_amount),
                    'payment_status'  => Transaction::COMPLETED,
                ];

                $transaction->fill($transactionData);
                $transaction->save();

                if($transaction->exists()){
                    DB::commit();
                    $res['status'] = 1;
                    $res['title'] = 'Success';
                    $res['message'] = "Paid Amount added successfully";
                }else{
                    $res['status'] = 0;
                    $res['title'] = 'Error';
                    $res['message'] = "Unable to save transaction for the paid amount!";
                }
            } else {
                $res['status'] = 0;
                $res['title'] = 'Error';
                $res['message'] = "Invoice not updated!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updatePaidAmount on InvoicesController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }
 
    public function cost($invoice_id)
    {
        $res = [];
        try {
            $invoice = Invoice::with(['invoice_item','invoice_item.gold_color','invoice_item.job.first_image','invoice_item.jewellery.first_image'])->where('id',$invoice_id)->firstOrFail();   
            return view('admin.invoices.invoice_cost',compact('invoice'));
        } catch (\Exception $err) {
            Log::error('Error in cost on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function updateCost(Request $request, $invoice_id)
    {
        $res = [];
        DB::beginTransaction();
        try {
                $invoice = Invoice::where('id',$invoice_id)->firstOrFail();
                $invoiceItem = collect($request->invoice);

                $price  = $invoiceItem->map(function($val){
                return $val['price'];
                });

                $subtotal =  array_sum($price->toArray());

                $settings  = Setting::price_settings();
                $gst_percent  = ($settings['gst_percent']) ? $settings['gst_percent'] : 0;
                $gst_amount = ($subtotal *  $gst_percent)/ 100;

                $invoice->layaway = $request->layaway;
                $invoice->sub_total = round($subtotal, 2);
                $invoice->gst_percent = round($gst_percent, 2);
                $invoice->gst_amount = $gst_amount ;
                $invoice->total = round(($subtotal + $gst_amount), 2)  ;
                $invoice->due =  $invoice->total;
                $invoice->save(); 

                if($invoice->exists()){
                foreach($request->invoice as $value){
                    $invoiceItem = InvoiceItem::where('id',$value['invoice_item_id'])->firstOrFail();
                    $invoiceItem->price = $value['price'];
                    $invoiceItem->sub_total = $value['price'];
                    $invoiceItem->save();      
                 }
                DB::commit();
                if(auth()->user()->hasPermissionTo('invoice-read'))
                {
                    return redirect()->route('admin.invoices.index')->with('success',"Invoice updated siccessfully!");
                } else{
                    return redirect()->route('admin.dashboard.index') ;
                } 
              } else{
                return back()->with('error', "Invoice not updated");
              }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in cost on InvoicesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }


}
