<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\SingleDropzoneComponent;
use App\Http\Requests\Admin\Customers\CustomersRequest;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Request;
use App\Mail\RegisterMail;
use App\Models\InvoiceItem;
use App\Models\Invoice;
use App\Models\User;
use Validator;
use File;
use Log;
use DB;

class CustomersController extends Controller
{

    function __construct(SingleDropzoneComponent $singleDropzone)
    {
        $this->singleDropzone = $singleDropzone;
    }

    public function index(Request $request, $tab = null)
    {
        try {
            if($tab == null){
                $tab = User::TAB_ALL;
            }else{
                $tab = $tab;
            }

            $all = $this->allCustomers($request);
            $vip = $this->allVipCustomers($request);
            return view('admin.customers.index', compact(['tab', 'all', 'vip']));
        } catch (\Exception $err) {
            Log::error('Error in index on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function allCustomers($request)
    {
        try{
            $records = User::where('status', User::ACTIVE)->sortable(['id' => 'desc']);
            $this->commonSearch($request, $records);
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return $records;
        }catch(\Exception $err){
            Log::error('Error in allCustomers on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function allVipCustomers($request)
    {
        try{
            $records = User::where(['status'=>User::ACTIVE, 'is_vip'=>User::ONE])->sortable(['id' => 'desc']);
            $this->commonSearch($request, $records);
            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return $records;
        }catch(\Exception $err){
            Log::error('Error in allVipCustomers on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function commonSearch($request, $records)
    {
        try {
            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('fullname', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('email', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere('phone', 'LIKE', '%' . $request->query('search') . '%');
                    $q->orWhere(DB::raw("concat(COALESCE(`country_code`,''),' ', phone)"), 'LIKE', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }
                });
            }
            return $records;
        }catch(\Exception $err){
            Log::error('Error in commonSearch on CustomerController :' . $err->getMessage());
        }
    }

    public function transactionsList(Request $request, $customer_id)
    {
        try {
            $customer = User::findOrFail($customer_id);

            $records = Invoice::with(['customer'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('invoice_no', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('invoice_no', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_name', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_email', 'like', '%' . $request->query('search') . '%');
                    $q->orWhere('customer_phone', 'like', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                        $q->orWhere('reference_date', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('customer', function ($query) use ($request) {
                        $query->where('fullname', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('email', 'like', '%' . $request->query('search') . '%');
                        $query->orWhere('phone', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.customers.transactions', compact(['customer', 'records']));
        }catch(\Exception $err){
            Log::error('Error in transactionsList on CustomerController :' . $err->getMessage());
        }
    }

    public function invoiceItemsList(Request $request, $customer_id, $invoice_id)
    {
        try {
            $records = InvoiceItem::with(['invoice', 'invoice.customer'])->where('invoice_id', $invoice_id)->sortable(['id' => 'desc']);

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
          
            return view('admin.customers.invoice_items', compact(['customer_id', 'invoice_id', 'records']));
        }catch(\Exception $err){
            Log::error('Error in invoiceItemsList on CustomerController :' . $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new User();
            return view('admin.customers.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(CustomersRequest $request)
    {
        try {
            DB::beginTransaction();
            $validated = $request->validated();

            if (!empty($validated['image'])) {
                $imagePath = User::IMAGE_PATH;
                $tmpDirPath = User::TEMP_PATH;
                $tablename = User::TABLENAME;
                $thumbImagePath = User::THUMB_IMAGE_PATH;
                $thumbSize = User::THUMB_SIZE;
                $this->singleDropzone->storeImage($validated, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);
            }
            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);
            
            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;
            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];
            $validated['status'] = User::ACTIVE;

            $customer = User::create($validated);
            if($customer->exists()){
                DB::commit();
                if(auth()->user()->hasPermissionTo('customer-read'))
                {
                    return redirect()->route('admin.customers.index')->with(['success' => 'Customer added successfully!']);
                } else{
                    return redirect()->route('admin.dashboard.index')->with('success', 'Customer added successfully!');
                } 
            }else{
                return back()->with(['error' => 'Customer not added!']);
            }  
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function edit($id)
    {
        try {
            $record = User::findOrFail($id);
            return view('admin.customers.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(CustomersRequest $request, $id)
    {
        try {
            $validated = $request->validated();
            $customer = User::findOrFail($id);
            $imagePath = User::IMAGE_PATH;
            $tmpDirPath = User::TEMP_PATH;
            $tablename = User::TABLENAME;
            $thumbImagePath = User::THUMB_IMAGE_PATH;
            $thumbSize = User::THUMB_SIZE;
            $validated['image'] = $this->singleDropzone->updateImage($validated, $customer, $imagePath, $tmpDirPath, $tablename, $thumbImagePath, $thumbSize);
            $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['phone']);
            $validated['dob'] = !empty($validated['dob']) ? date('Y-m-d', strtotime($validated['dob'])) : null;
            $validated['country_code'] = $phoneInputRes['country_code'];
            $validated['phone'] = $phoneInputRes['phone'];

            $customer->fill($validated);
            $customer->save();
            if( $customer->exists()){
                if(auth()->user()->hasPermissionTo('customer-read'))
                {
                  return redirect()->route('admin.customers.index')->with(['success' => 'Customer updated successfully!']);
                } else{
                return redirect()->route('admin.dashboard.index')->with('success', 'Customer added successfully!');
                } 
            } 
        } catch (\Exception $err) {
            Log::error('Error in update on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }  
    }

    public function destroy($id)
    {
        try {
            $record = User::findOrFail($id);
            if ($record->delete()) {
                if ($record->image) {
                    $imagePath =   User::IMAGE_PATH;
                    $thumbImagePath =  User::THUMB_IMAGE_PATH;
                    $this->singleDropzone->deleteImage($record,$imagePath,$thumbImagePath);
                }
                return back()->with(['success' => 'customer deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record!']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on CustomerController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function makeVIP(Request $request)
    {
        $res = [];
        try {
            $validator = Validator::make($request->all(), [ 
                'customer_id' => "required|integer|not_in:0",
                'is_vip'      => "required",
            ],[
                'customer_id.required' => "Customer id is required.",
                'is_vip.required' => "Vip field is required.",
            ]);
             
            if ($validator->fails()) {
                $res['error'] = 1;
                $res['message'] = $validator->errors()->first();
                return response()->json($res);
            }
            $record = User::findOrFail($request->customer_id);
            $record->is_vip = $request->is_vip;

            $dynamicMessage = ($request->is_vip == 1) ? 'Cutomer is added to'  : 'Cutomer is removed from';

            $record->save();
            if ($record->wasChanged()) {
                $res['error'] = 0;
                $res['message'] = $dynamicMessage.' <strong>VIP Group</strong>.';
            } else {
                $res['error'] = 1;
                $res['message'] = 'Unable to make change in customer!';
            }
        } catch (\Exception $err) {
            Log::error('Error in makeVIP on CustomerController :' . $err->getMessage());
            $res['error'] = 2;
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }
}
