<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Components\CustomerComponent;
use Illuminate\Database\Eloquent\Collection;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\Admin\Consignments\ConsignmentsRequest;
use App\Http\Requests\Admin\Consignments\ConsignmentReviewCostsRequest;
use App\Http\Requests\Admin\Consignments\UpdateUsedAmountRequest;
use Illuminate\Support\Facades\Mail;
use App\Mail\RegisterMail;
use App\Helpers\CommonHelper;
use App\Models\ConsignmentReviewCost;
use App\Models\JewelleryLoosePiece;
use App\Models\JewelleryImage;
use App\Models\JobLoosePiece;
use App\Models\ReviewCost;
use App\Models\ConsignmentItem;
use App\Models\Consignment;
use App\Models\Jewellery;
use App\Models\JobImage;
use App\Models\Category;
use App\Models\User;
use App\Models\Type;
use App\Models\Job;
use File;
use Log;
use DB;

class ConsignmentsController extends Controller
{
    const ZEROES = 5;
    const CONSIGNMENT = "Consignment";

    public function __construct()
    {
        $this->CustomerComponent = new CustomerComponent();
    }

    public function index(Request $request)
    {
        try {
            $records = Consignment::with(['consignment_item', 'consignment_item.category', 'consignment_review_cost'])->sortable(['id' => 'desc']);

            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('consignment_no', 'LIKE', '%' . $request->query('search') . '%');

                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                        $q->orWhere('reference_date', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }

                    $q->orWhereHas('consignment_item.category', function ($query) use ($request) {
                        $query->where('name', 'like', '%' . $request->query('search') . '%');
                    });
                });
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));
            return view('admin.consignments.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Consignment();
            return view('admin.consignments.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(ConsignmentsRequest $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);
            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customerData, self::CONSIGNMENT);
            if ($customerRes['status'] == 1) {
                $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['customer_phone']);
                $validated['country_code'] = $phoneInputRes['country_code'];
                $validated['customer_phone'] = $phoneInputRes['phone'];
                $validated['customer_id'] = $customerRes['data'];
                $validated['reference_date'] = date('Y-m-d', strtotime($validated['reference_date']));
                $validated['used'] = 0;
                $validated['remainder'] = $validated['weight'];
                $consignment = Consignment::create($validated);

                if ($consignment->exists()) {
                    $consignment_id = $consignment->id;
                    $updateConsignmentRes = $this->_updateConsignmentNo($consignment_id);
                    if ($updateConsignmentRes['status'] == 1) {
                        $total_price = $validated['weight'] * $validated['unit_price'];
                        $data = $validated;
                        $data['consignment_id'] = $consignment_id;
                        $data['total_price'] = $total_price;
                        $consignment_item = ConsignmentItem::create($data);
                        if ($consignment_item->exists()) {
                            DB::commit();
                                return redirect()->route('admin.consignments.reviewCostSummary', $consignment_id)->with('success', 'Consignment added successfully.'); 
                        } else {
                            return back()->with('error', 'Unable to add consignment item!');
                        }
                    } else {
                        return back()->with('error', $updateConsignmentRes['message']);
                    }
                } else {
                    return back()->with('error', 'Unable to add consignment!');
                }
            } else {
                return back()->with('error', $customerRes['message']);
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in store on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function _updateConsignmentNo($consignment_id)
    {
        $res = [];
        try {
            $record = Consignment::findOrFail($consignment_id);
            $record->consignment_no = getItemCode(self::ZEROES, $consignment_id, self::CONSIGNMENT);
            $record->save();
            if ($record->wasChanged()) {
                $res['status'] = 1;
                $res['message'] = "Consignment no. updated successfully.";
                $res['data'] = $record->consignment_no;
            } else {
                $res['status'] = 0;
                $res['message'] = "Consignment no. not updated!";
            }
        } catch (\Exception $err) {
            Log::error('Error in _updateConsignmentNo on ConsignmentsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['message'] = $err->getMessage();
        }
        return $res;
    }

    public function edit($id)
    {
        try {
            $record = Consignment::findOrFail($id);
            return view('admin.consignments.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(ConsignmentsRequest $request, $consignment_id)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $customerData = $request->only(['customer_name', 'customer_email', 'customer_phone']);

            $consignment = Consignment::with('consignment_item')->findOrFail($consignment_id);

            $customerRes = $this->CustomerComponent->addOrupdateCustomer($customerData, self::CONSIGNMENT);
            if ($customerRes['status'] !== 1) {
                return back()->with('error', $customerRes['message']);
            }else{
                $phoneInputRes = $this->separateDetailsFromPhoneInput($validated['customer_phone']);
                $validated['country_code'] = $phoneInputRes['country_code'];
                $validated['customer_phone'] = $phoneInputRes['phone'];
            }

            if (!($consignment->consignment_no)) {
                $updateConsignmentRes = $this->_updateConsignmentNo($consignment_id);
                if ($updateConsignmentRes['status'] !== 1) {
                    return back()->with('error', $updateConsignmentRes['message']);
                } else {
                    $validated['consignment_no'] = $updateConsignmentRes['data'];
                }
            }

            $validated['customer_id'] = $customerRes['data'];
            $validated['reference_date'] = date('Y-m-d', strtotime($validated['reference_date']));

            $consignment->fill($validated);
            $consignment->save();
            if ($consignment->exists()) {
                $total_price = $validated['weight'] * $validated['unit_price'];
                $data = $validated;
                $data['consignment_id'] = $consignment_id;
                $data['total_price'] = $total_price;

                $consignment_item = ConsignmentItem::where('consignment_id', $consignment_id)->firstOrFail();
                $consignment_item->fill($data);
                $consignment_item->save();
                if ($consignment_item->exists()) {
                    DB::commit();
                        return redirect()->route('admin.consignments.reviewCostSummary', $consignment_id)->with('success', 'Consignment updated successfully.');
                } else {
                    return back()->with('error', 'Consignment item not updated!');
                }
            } else {
                return redirect()->route('admin.consignments.reviewCostSummary', $consignment_id)->with('error', 'Consignment not updated!');
            }
            DB::commit();
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in update on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function reviewCostSummary($id)
    {
        try {
            $record = Consignment::with('consignment_item')->withCount(['consignment_item as subtotal' => function ($q) {
                $q->select(DB::raw("SUM(total_price)"));
            }])->findOrFail($id);
            return view('admin.consignments.review_cost_summary', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in repairCostSummary on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function updateCost(ConsignmentReviewCostsRequest $request)
    {
        try {
            $consignment = Consignment::findOrFail($request->consignment_id);
            $consignment->reference_date = date('Y-m-d', strtotime($request->reference_date));
            $consignment->save();
            if ($consignment) {
                $inputData = $request->except('reference_date');
                $consignmentReviewCost = ConsignmentReviewCost::firstOrNew(['consignment_id' => $inputData['consignment_id']]);
                $consignmentReviewCost->fill($inputData);
                $consignmentReviewCost->save();
                if ($consignmentReviewCost->exists() && !$consignmentReviewCost->wasChanged()) { // $consignmentReviewCost->wasRecentlyCreated()
                    if(auth()->user()->hasPermissionTo('consignment-read'))
                    {
                        return redirect()->route('admin.consignments.index')->with(['success' => 'Review Cost added successfully!']);
                    } else{
                    return redirect()->route('admin.dashboard.index')->with('success', 'Review Cost added successfully!');
                    }   
                } elseif ($consignmentReviewCost->exists() && $consignmentReviewCost->wasChanged()) {
                    if(auth()->user()->hasPermissionTo('consignment-read'))
                    {
                        return redirect()->route('admin.consignments.index')->with(['success' => 'Review Cost update successfully!']);
                    } else{
                    return redirect()->route('admin.dashboard.index')->with('success', 'Review Cost update successfully!');
                    }
                } else {
                    return redirect()->route('admin.consignments.index')->with(['error' => 'Something went wrong.']);
                }
            }
        } catch (\Exception $err) {
            Log::error('Error in updateCost on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Consignment::findOrFail($id);
            if ($record->delete()) {
                return back()->with(['success' => 'Consignment deleted successfully!']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function printReceipt($consignment_id)
    {
        try {
            $consignment = Consignment::with(['consignment_item', 'consignment_item.category', 'consignment_review_cost'])->findOrFail($consignment_id);
            return view('admin.consignments.print.receipt', compact('consignment'));
        } catch (\Exception $err) {
            Log::error('Error in printReceipt on ConsignmentsController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function updateAmountUsed(UpdateUsedAmountRequest $request)
    {
        $res = [];
        try {
            $validated = $request->validated();
            $record = ConsignmentItem::where(['id'=>$validated['consignment_item_id'], 'consignment_id'=>$validated['consignment_id']])->firstOrFail();

            $data = [
                'used'      => $validated['used'],
                'remainder' => round(($record->weight - $validated['used']), 2),
            ];

            $record->fill($data);
            $record->save();
            if ($record->wasChanged()) {  
                $res['status'] = 1;
                $res['title'] = 'Success';
                $res['message'] = "Amount used updated successfully!!";
                $res['data'] = $record;
            } else {
                $res['status'] = 0;
                $res['title'] = 'Error';
                $res['message'] = "Unable to update amount used!";
            }
        } catch (\Exception $err) {
            DB::rollback();
            Log::error('Error in updateAmountUsed on ConsignmentsController :' . $err->getMessage());
            $res['status'] = 0;
            $res['title'] = 'Error';
            $res['message'] = $err->getMessage();
        }
        return response()->json($res);
    }

}
