<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\CategoriesRequest;
use Illuminate\Http\Request;
use App\Models\Category;
use File;
use DB;
use Log;

class CategoriesController extends Controller
{

    public function index(Request $request)
    {
        try {
            $records = Category::sortable(['id' => 'desc']);
            if ($request->query('search')) {
                $records = $records->where(function ($q) use ($request) {
                    $q->where('name', 'LIKE', '%' . $request->query('search') . '%');
                    $search_array = explode('/', $request->query('search'));
                    if (count($search_array) !== 1) {
                        [$search_array[0], $search_array[1]] = [$search_array[1], $search_array[0]];
                        $new_date = implode('/', $search_array);
                        $q->orWhere('created_at', 'LIKE', '%' . date('Y-m-d', strtotime($new_date)) . '%');
                    }
                });
            }

            if ($request->query('start_date') || $request->query('end_date')) {
                $start_date = (!empty($request->start_date)) ? \Carbon\Carbon::parse($request->start_date)->toDateTimeString() : \Carbon\Carbon::now();
                $end_date = (!empty($request->end_date)) ? \Carbon\Carbon::parse($request->end_date)->toDateTimeString() : \Carbon\Carbon::now();
                $records = $records->whereBetween('created_at', [$start_date, $end_date]);
            }

            $records = $records->paginate(($request->query('limit') ? $request->query('limit') : env('PAGINATION_LIMIT')));

            return view('admin.categories.index', compact('records'));
        } catch (\Exception $err) {
            Log::error('Error in index on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function create()
    {
        try {
            $record = new Category();
            return view('admin.categories.create', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in create on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function store(CategoriesRequest $request)
    {
        try {
            $validated = $request->validated();
            $validated['name'] = ucwords($validated['name']);

            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $ext = $file->getClientOriginalExtension();

                if (!File::exists(public_path(Category::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Category::IMAGE_PATH));
                }
                $file_name = 'category' . time() . "." . $ext;

                if ($file->move(public_path(Category::IMAGE_PATH), $file_name)) {
                    $validated['image'] = $file_name;
                } else {
                    return back()->with('error', "Unable to upload image.");
                }
            }

            if ($request->hasFile('icon_image')) {
                $file_icon = $request->file('icon_image');
                $ext_icon = $file_icon->getClientOriginalExtension();

                if (!File::exists(public_path(Category::IMAGE_PATH))) {
                    File::makeDirectory(public_path(Category::IMAGE_PATH));
                }
                $file_icon_name = 'category_icon' . time() . "." . $ext_icon;

                if ($file_icon->move(public_path(Category::IMAGE_PATH), $file_icon_name)) {
                    $validated['icon_image'] = $file_icon_name;
                } else {
                    return back()->with('error', "Unable to upload image.");
                }
            }

            $categories = Category::create($validated);
            return redirect()->route('admin.categories.index')->with(['success' => 'category added successfully.']);
        } catch (\Exception $err) {
            Log::error('Error in store on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function edit($id)
    {
        try {
            $record = Category::findOrFail($id);
            return view('admin.categories.edit', compact('record'));
        } catch (\Exception $err) {
            Log::error('Error in edit on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function update(CategoriesRequest $request, $id)
    {
        $validated = $request->validated();
        $category = Category::findOrFail($id);
        //////image //////
        $oldImage = $category->image;
        $oldImageArray = explode('/', $oldImage);
        $oldImageName = end($oldImageArray);
        if ($request->hasFile('image')) {
            if (isset($oldImageName)) {
                $old_image_path = public_path(Category::IMAGE_PATH) . '/' . $oldImageName;
                if (File::exists($old_image_path)) {
                    File::delete($old_image_path);
                }
            }
            $file = $request->file('image');
            $ext = $file->getClientOriginalExtension();
            $file_name = "category" . time() . "." . $ext;
            if ($file->move(public_path(Category::IMAGE_PATH), $file_name)) {
                $validated['image'] = $file_name;
            } else {
                return redirect()->route('admin.categories.index')->with(['error' => 'Unable to upload image.']);
            }
        }

        //////image Icon ///////////
        $oldImageIcon = $category->icon_image;
        $oldImageIconArray = explode('/', $oldImageIcon);
        $oldImageIconName = end($oldImageIconArray);
        if ($request->hasFile('icon_image')) {
            if (isset($oldImageIconName)) {
                $old_image_icon_path = public_path(Category::IMAGE_PATH) . '/' . $oldImageIconName;
                if (File::exists($old_image_icon_path)) {
                    File::delete($old_image_icon_path);
                }
            }
            $file_icon = $request->file('icon_image');
            $ext_icon = $file_icon->getClientOriginalExtension();
            $file_icon_name = 'category_icon' . time() . "." . $ext_icon;
            if ($file_icon->move(public_path(Category::IMAGE_PATH), $file_icon_name)) {
                $validated['icon_image'] = $file_icon_name;
            } else {
                return redirect()->route('admin.categories.index')->with(['error' => 'Unable to upload image.']);
            }
        }
        $category->fill($validated);
        $category->save();
        return redirect()->route('admin.categories.index')->with(['success' => 'Category updated successfully.']);
    }

    public function changeStatus(Request $request)
    {
        try {
            $record = Category::findOrFail($request->id);
            $record->status = $request->status;
            if ($record->save()) {
                $error = 0;
                $message = 'Status changed to <strong>' . $record->status . '</strong>.';
            } else {
                $error = 1;
                $message = 'Unable to change status.';
            }
            return response()->json([
                'error' => $error,
                'message' => $message
            ]);
        } catch (\Exception $err) {
            Log::error('Error in changeStatus on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $record = Category::findOrFail($id);

            if ($record->delete()) {
                if ($record->image) {
                    $oldImage = $record->image;
                    $oldImageArray = explode('/', $oldImage);
                    $oldImageName = end($oldImageArray);
                    $old_image_path = public_path(Category::IMAGE_PATH) . '/' . $oldImageName;
                    if (File::exists($old_image_path)) {
                        File::delete($old_image_path);
                    }
                }

                if ($record->icon_image) {
                    $oldImageIcon = $record->icon_image;
                    $oldImageIconArray = explode('/', $oldImageIcon);
                    $oldImageIconName = end($oldImageIconArray);
                    $old_image_icon_path = public_path(Category::IMAGE_PATH) . '/' . $oldImageIconName;
                    if (File::exists($old_image_icon_path)) {
                        File::delete($old_image_icon_path);
                    }
                }
                
                return back()->with(['success' => 'Category deleted successfully.']);
            } else {
                return back()->with(['error' => 'Unable to delete this record.']);
            }
        } catch (\Exception $err) {
            Log::error('Error in destroy on CategoriesController :' . $err->getMessage());
            return back()->with('error', $err->getMessage());
        }
    }
}
